/* date.c
   **
   ** released into the PUBLIC DOMAIN 30 jul 1990 by jim nutt
   ** Changes released into the PUBLIC DOMAIN 10 jul 1994 by John Dennis
   **
   ** Parse various string date formats into a unix style timestamp.
   **
 */

#include "msged.h"
#include "date.h"
#include "strextra.h"

#define valid_date(timestamp) \
       !((timestamp->tm_wday > 6)  || (timestamp->tm_wday < 0) || \
     (timestamp->tm_mon > 11)  || (timestamp->tm_mon < 0)  || \
     (timestamp->tm_mday > 31) || (timestamp->tm_mday < 0) || \
     (timestamp->tm_year > 99) || (timestamp->tm_year < 0) || \
     (timestamp->tm_hour > 23) || (timestamp->tm_hour < 0) || \
     (timestamp->tm_min > 59)  || (timestamp->tm_min < 0)  || \
     (timestamp->tm_sec > 59)  || (timestamp->tm_sec < 0))

static char *month[] =
{
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

static char *day[] =
{
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

static char *attr_tokens[] =
{
    "yms",                      /* year msg */
    "yno",                      /* year now */
    "mms",                      /* month msg */
    "mno",                      /* month now */
    "dms",                      /* day msg */
    "dno",                      /* day now */
    "wms",                      /* weekday msg */
    "wno",                      /* weekday now */
    "tnm",                      /* time msg normal */
    "tnn",                      /* time now normal */
    "tam",                      /* time msg atime */
    "tan",                      /* time now atime */

    /* ------------- */

    "ofn",                      /* orginal from name */
    "off",                      /* original from first name */
    "otn",                      /* original to name */
    "otf",                      /* original to first name */
    "osu",                      /* original subject */
    "ooa",                      /* original origin address */
    "oda",                      /* orginal destination address */

    /* ------------- */

    "fna",                      /* from name */
    "ffn",                      /* from first name */
    "fad",                      /* from address */
    "tna",                      /* to name */
    "tfn",                      /* to first name */
    "tad",                      /* to address */
    "sub",                      /* subject */

    /* ------------- */

    "una",                      /* user name */
    "ufn",                      /* user first name */
    "uad",                      /* user address */
    "ceh",                      /* current echo tag */
    "oeh",                      /* old echo tag */

    "ims",                      /* iso date of msg */
    "ino",                      /* iso date now */
    "cms",                      /* 4-digit year of msg */
    "cno",                      /* 4-digit year of now */
    NULL
};

time_t parsedate(char *ds)
{
    char work[80];
    char *s;
    int t;
    struct tm timestamp;

    if (ds == NULL || strlen(ds) == 0)
        return 0;

    memset(&timestamp, 0, sizeof timestamp);
    strcpy(work, ds);

    if (strchr(ds, '-') != NULL)
    {                           /* quickbbs style date */
        s = strtok(work, "-");
        timestamp.tm_mon = atoi(s) - 1;
        s = strtok(NULL, "-");
        timestamp.tm_mday = atoi(s);
        s = strtok(NULL, " ");
        timestamp.tm_year = atoi(s);
        s = strtok(NULL, ":");
        while (isspace(*s))
            s++;
        timestamp.tm_hour = atoi(s);
        s = strtok(NULL, " ");
        timestamp.tm_min = atoi(s);
    }
    else
    {                           /* fido style date */
        s = strtok(work, " ");

        if (s == NULL)
            return 0;

        if ((t = atoi(s)) == 0)
        {                       /* a usenet date */
            s = strtok(NULL, " ");
            if (s == NULL)
                return 0;
            t = atoi(s);
        }
        timestamp.tm_mday = t;
        s = strtok(NULL, " ");
        if (s == NULL)
            return 0;
        for (t = 0; t < 12; t++)
            if (stricmp(s, month[t]) == 0)
                break;
        if (t == 12)
            t = 1;              /* WRA */
        timestamp.tm_mon = t;
        s = strtok(NULL, " ");
        if (s == NULL)
            return 0;
        timestamp.tm_year = atoi(s);
        s = strtok(NULL, ":");
        if (s == NULL)
            return 0;
        while (isspace(*s))
            s++;
        timestamp.tm_hour = atoi(s);
        s = strtok(NULL, ": \0");
        if (s == NULL)
            return 0;
        timestamp.tm_min = atoi(s);
        s = strtok(NULL, " ");
        if (s != NULL)
            timestamp.tm_sec = atoi(s);
        timestamp.tm_isdst = -1;
    }
#ifdef NEVER
    timestamp.tm_hour += 5;
#endif
    return mktime(&timestamp);
}

char *itime(time_t now)
{
    static char itime_buffer[40];
    struct tm *timestamp;

    timestamp = localtime(&now);

    if (!timestamp || !valid_date((timestamp)))
        return ("invalid date");

    sprintf(itime_buffer, "%04d-%02d-%02d %02d:%02d:%02d",
            timestamp->tm_year + 1900, timestamp->tm_mon + 1,
            timestamp->tm_mday, timestamp->tm_hour,
            timestamp->tm_min, timestamp->tm_sec);
    return (itime_buffer);
}

char *atime(time_t now)
{
    static char atime_buffer[40];
    struct tm *timestamp;

    timestamp = localtime(&now);

    if (!timestamp || !valid_date((timestamp)))
        return ("invalid date");

    sprintf(atime_buffer, "%s %s %02d %04d %02d:%02d:%02d",
            day[timestamp->tm_wday], month[timestamp->tm_mon],
            timestamp->tm_mday, timestamp->tm_year + 1900,
            timestamp->tm_hour, timestamp->tm_min, timestamp->tm_sec);
    return (atime_buffer);
}

char *mtime(time_t now)
{
    static char mtime_buffer[21];
    struct tm *timestamp;

    timestamp = localtime(&now);

    if (!timestamp || !valid_date((timestamp)))
        return ("invalid date");

    sprintf(mtime_buffer, "%02d %s %02d  %02d:%02d:%02d",
            timestamp->tm_mday, month[timestamp->tm_mon],
            timestamp->tm_year, timestamp->tm_hour,
            timestamp->tm_min, timestamp->tm_sec);
    return (mtime_buffer);
}

char *qtime(time_t now)

{
    static char qtime_buffer[20];
    struct tm *timestamp;

    timestamp = localtime(&now);

    if (!timestamp || !valid_date((timestamp)))
        return ("invalid date");

    sprintf(qtime_buffer, "%s %02d %02d:%02d",
            month[timestamp->tm_mon], timestamp->tm_mday,
            timestamp->tm_hour, timestamp->tm_min);
    return (qtime_buffer);
}

/*
   **
   ** Returns the token number or -1 if not found.
   **
 */

int find_token(char *token)
{
    int i;

    i = 0;
    while (attr_tokens[i] != NULL)
    {
        if (stricmp(attr_tokens[i], token) == 0)
            return i;
        i++;
    }
    return -1;
}

/*
   **
   ** Returns a pointer to the first name - NOTE uses static memory.
   **
 */

char *firstname(char *name)
{
    static char work[40];
    char *s;

    memset(work, 0, sizeof(work));
    if ((s = strchr(name, ' ')) == NULL)
        strcpy(work, name);
    else
    {
        *s = '\0';
        strcpy(work, name);
        *s = ' ';
    }
    return work;
}

/*
   **
   ** Builds an attribution line.
   **
 */

char *attrib_line(msg * m, msg * old, int olda, char *format)
{
    struct tm now;
    struct tm *timestamp;
    char work[256];
    char token[5];
    char *t;
    time_t n;
    int num;

    if (format == NULL)
        return NULL;

    memset(work, 0, sizeof(work));
    t = work;
    n = time(NULL);
    timestamp = localtime(&n);
    now = *timestamp;

    if (old)
        timestamp = localtime(&(old->timestamp));

    while (*format)
    {
        if (*format == '%')
        {
            format++;
            switch (*format)
            {
            case '%':
                *t = *format;
                break;

            case '_':
                *t = ' ';
                break;

            default:
                memset(token, 0, sizeof(token));
                strncpy(token, format, 3);
                num = find_token(token);

                switch (num)
                {
                case 0:
                    if (old)
                        sprintf(t, "%02d", timestamp->tm_year);
                    break;
                case 1:
                    sprintf(t, "%02d", now.tm_year);
                    break;
                case 2:
                    if (old)
                        strcpy(t, month[timestamp->tm_mon]);
                    break;
                case 3:
                    strcpy(t, month[now.tm_mon]);
                    break;
                case 4:
                    if (old)
                        sprintf(t, "%02d", timestamp->tm_mday);
                    break;
                case 5:
                    sprintf(t, "%02d", now.tm_mday);
                    break;
                case 6:
                    if (old)
                        strcpy(t, day[timestamp->tm_wday]);
                    break;
                case 7:
                    strcpy(t, day[now.tm_wday]);
                    break;
                case 8:
                    if (old)
                        sprintf(t, "%02d:%02d", timestamp->tm_hour, timestamp->tm_min);
                    break;
                case 9:
                    sprintf(t, "%02d:%02d", now.tm_hour, now.tm_min);
                    break;
                case 10:
                    if (old)
                        strcpy(t, atime(old->timestamp));
                    break;
                case 11:
                    strcpy(t, atime(n));
                    break;
                case 12:
                    if (old && old->isfrom)
                        strcpy(t, old->isfrom);
                    break;
                case 13:
                    if (old && old->isfrom)
                        strcpy(t, firstname(old->isfrom));
                    break;
                case 14:
                    if (old && old->isto)
                        strcpy(t, old->isto);
                    break;
                case 15:
                    if (old && old->isto)
                        strcpy(t, firstname(old->isto));
                    break;
                case 16:
                    if (old && old->subj)
                        strcpy(t, old->subj);
                    break;
                case 17:
                    if (old)
                        strcpy(t, show_address(&old->from));
                    break;
                case 18:
                    if (old)
                        strcpy(t, show_address(&old->to));
                    break;
                case 19:
                    if (m->isfrom)
                        strcpy(t, m->isfrom);
                    break;
                case 20:
                    if (m->isfrom)
                        strcpy(t, firstname(m->isfrom));
                    break;
                case 21:
                    strcpy(t, show_address(&m->from));
                    break;
                case 22:
                    if (m->isto)
                        strcpy(t, m->isto);
                    break;
                case 23:
                    if (m->isto)
                        strcpy(t, firstname(m->isto));
                    break;
                case 24:
                    strcpy(t, show_address(&m->to));
                    break;
                case 25:
                    strcpy(t, m->subj);
                    break;
                case 26:
                    if (ST->username)
                        strcpy(t, ST->username);
                    break;
                case 27:
                    if (ST->username)
                        strcpy(t, firstname(ST->username));
                    break;
                case 28:
                    strcpy(t, show_address(&CurArea.addr));
                    break;
                case 29:
                    if (CurArea.netmail)
                        strcpy(t, "netmail");
                    else
                        strcpy(t, CurArea.tag);
                    break;
                case 30:
                    if (olda != -1)
                    {
                        if (arealist[olda].netmail)
                            strcpy(t, "netmail");
                        else
                            strcpy(t, arealist[olda].tag);
                    }
                    break;
                case 31:
                    if (old != NULL)
                    {
                        sprintf(t,
                                "%04d-%02d-%02d",
                                timestamp->tm_year + 1900,
                                timestamp->tm_mon + 1,
                                timestamp->tm_mday);
                    }
                    break;
                case 32:
                    sprintf(t,
                            "%04d-%02d-%02d",
                            now.tm_year + 1900,
                            now.tm_mon + 1,
                            now.tm_mday);
                    break;
                case 33:
                    if (old != NULL)
                    {
                        sprintf(t, "%04d", timestamp->tm_year + 1900);
                    }
                    break;
                case 34:
                    sprintf(t, "%04d", now.tm_year + 1900);
                    break;
                default:
                    break;
                }
                break;
            }
            t = work + strlen(work);
            format += 3;
        }
        else if (*format == '\\')
        {
            if (*(++format) == 'n')
                *t++ = '\n';
            format++;
        }
        else
            *t++ = *format++;
    }
    return strdup(work);
}

char *attrib_lineold(msg * m, char *format)
{
    char work[256];
    char *t = work;
    char *s;
    struct tm *timestamp;
    struct tm now;
    time_t n = time(NULL);

    if (format == NULL)
        return (NULL);

    memset(work, 0, sizeof work);
    timestamp = localtime(&n);
    memcpy(&now, timestamp, sizeof(struct tm));
    timestamp = localtime(&(m->timestamp));

    while (*format)
    {
        if (*format == '%')
        {
            format++;
            switch (tolower(*format))
            {
            case '%':
                *t = *format;
            default:
                break;
            case 't':
                if (m->isto)
                    strcpy(t, m->isto);
                break;
            case '_':
                strcpy(t, " ");
                break;
            case 'f':
                if (m->isfrom)
                    strcpy(t, m->isfrom);
                break;
            case 'i':
                if (m->isfrom)
                {
                    if ((s = strchr(m->isfrom, ' ')) == NULL)
                        strcpy(t, m->isfrom);
                    else
                    {
                        *s = '\0';
                        strcpy(t, m->isfrom);
                        *s = ' ';
                    }
                }
                break;
            case 's':
                if (m->isto)
                {
                    if ((s = strchr(m->isto, ' ')) == NULL)
                        strcpy(t, m->isto);
                    else
                    {
                        *s = '\0';
                        strcpy(t, m->isto);
                        *s = ' ';
                    }
                }
                break;

            case 'e':
                if (CurArea.echomail)
                    strcpy(t, CurArea.tag);
                else if (CurArea.netmail)
                    strcpy(t, "netmail");
                break;
            case 'a':
                strcpy(t, show_address(&m->from));
                break;
            case 'g':
                strcpy(t, show_address(&m->to));
                break;
            case 'w':
                format++;
                if (tolower(*format) == 'm')
                    strcpy(t, day[timestamp->tm_wday]);
                else
                    strcpy(t, day[now.tm_wday]);
                break;
            case 'd':
                format++;
                if (tolower(*format) == 'm')
                    sprintf(t, "%02d", timestamp->tm_mday);
                else
                    sprintf(t, "%02d", now.tm_mday);
                break;
            case 'y':
                format++;
                if (tolower(*format) == 'm')
                    sprintf(t, "%02d", timestamp->tm_year);
                else
                    sprintf(t, "%02d", now.tm_year);
                break;
            case 'm':
                format++;
                if (tolower(*format) == 'm')
                    strcpy(t, month[timestamp->tm_mon]);
                else
                    strcpy(t, month[now.tm_mon]);
                break;
            case 'h':
                format++;
                if (tolower(*format) == 'm')
                    sprintf(t, "%02d:%02d", timestamp->tm_hour, timestamp->tm_min);
                else
                    sprintf(t, "%02d:%02d", now.tm_hour, now.tm_min);
                break;
            case 'n':
                strcpy(t, atime(n));
                break;
            case 'u':
                if (ST->username)
                    strcpy(t, ST->username);
                break;
            case 'o':
                strcpy(t, show_address(&CurArea.addr));
                break;
            }
            t = work + strlen(work);
            format++;
        }
        else if (*format == '\\')
        {
            if (*(++format) == 'n')
                *t++ = '\n';
            format++;
        }
        else
            *t++ = *format++;
    }
    return strdup(work);
}

/* end of file */
