/*
 * This version of crypt() is optimized for use with John the Ripper by Solar
 * Designer, with some great hints from Roman Rusakov. Actually, it does not
 * support the standard crypt(3) interface -- it wouldn't be wise to deal with
 * the 13 character ASCII encrypted passwords in a brute force cracker. This
 * code is based on "crack-fcrypt.c" from Crack v4.1 by Alec Muffett, the time
 * critical routines are re-coded, while some of the initialization routines
 * are left unchanged. The original Alec Muffett's copyright notice follows:
 *
 * This program is copyright Alec Muffett 1991 except for some portions of
 * code in "crack-fcrypt.c" which are copyright Robert Baldwin, Icarus Sparry
 * and Alec Muffett.  The author(s) disclaims all responsibility or liability
 * with respect to its usage or its effect upon hardware or computer
 * systems, and maintain copyright as set out in the "LICENCE" document which
 * accompanies distributions of Crack v4.0 and upwards.
 */

#include <string.h>
#define bzero(buf, cnt) memset(buf, 0, cnt)

#define	reg	register
#define	uns	unsigned
#define unsb	uns char
#define unsw    uns short
#define	unsl	uns long

/*
 * Types for the different ways to represent DES bit patterns.  Bits are
 * always right justified within fields.  Bits which have lower indices in
 * the NBS spec are stored in the vax bits with less significance (e.g., Bit
 * 1 of NBS spec is stored in the bit with weight 2 ** 0 to the Vax.
 */

#define	obpb1	unsb		/* One bit per byte. */
#define sbpb6	unsb		/* Six bits per byte, 6 held. */
#define sbpb6R	unsb		/* Six bits per byte Reversed order, 6 held. */
#define	sbpb24	unsl		/* Six bits per byte, 24/48 held. */
#define	fbpb4	unsb		/* Four bits per byte, 4 held. */
#define	fbpb4R	unsb		/* Four bits per byte Reversed order, 4 held. */

#ifdef CRYPT_ASM
#define CRYPT_EXTERN
#endif

#ifdef CRYPT_EXTERN
#define EXTERN extern
#else
#define EXTERN
#endif

#ifdef CRYPT_64BIT
#define SIZEDIV         2
#else
#define SIZEDIV         1
#endif

#ifndef CRYPT_128K
/* DES transformation type... */
union SDATA
{
    sbpb24 b[2 / SIZEDIV];
    sbpb6 c[8];
};
#endif

#if defined(CRYPT_64BIT) || defined(CRYPT_MIXED)
#ifdef CRYPT_128K
#define SIZEFIX         3
#else
/*
 * It is not possible to shift by 3 (no free bits), so we have to choose
 * between 0 and 2 - SD.
 */
#if CRYPT_SCALE == 8
#define SIZEFIX         0
#else
#define SIZEFIX         2
#endif
#endif
#else
#define SIZEFIX         2
#endif

#ifdef CRYPT_128K
#if SIZEFIX == 3
#define DO_SIZEFIX(x)   (((((x) << 2) & 0x00FF00FF) | ((x) & 0xFF00FF00)) << 1)
#else
#define DO_SIZEFIX(x)   ((((x) << SIZEFIX) & 0x00FF00FF) | ((x) & 0xFF00FF00))
#endif
#else
#define DO_SIZEFIX(x)   ((x) << SIZEFIX)
#endif

/*
 * Some machine-dependent definitions follow. Be sure to use the correct
 * options while compiling, this might affect the performance a lot - SD.
 */

#if (defined(CRYPT_64BIT) && (defined(CRYPT_SCALE) || !defined(CRYPT_MASK)) || defined(CRYPT_MIXED)) && SIZEFIX == 2
/*
 * 64 bit architectures which can shift addresses left by 1 bit with no extra
 * time required (for example by adding a register to itself).
 */
#define INDIRECT(a, b)  (*((sbpb24 *)(((unsigned char (*)[2]) a) + (b))))
#else
#if SIZEFIX == 0
/*
 * 64 bit architectures which can shift addresses left by 3 bits (but maybe
 * not by 1) with no extra time required (for example by using the S8ADDQ
 * instruction on DEC Alphas; we would need an ADDQ anyway).
 */
#define INDIRECT(a, b)  a[b]
#else
/*
 * Architectures with no complicated addressing modes supported, or when those
 * are not required.
 */
#define INDIRECT(a, b)  (*((sbpb24 *)(((unsigned char *) a) + (b))))
#endif
#endif

/*
 * You can choose between using shifts/masks, and using a walking pointer.
 */

#ifdef CRYPT_MASK
#if defined(CRYPT_64BIT) && !defined(CRYPT_SCALE) && SIZEFIX == 2
/*
 * This method might be good for some 64 bit architectures with no complicated
 * addressing modes supported. It would be the best one for DEC Alphas if they
 * didn't have the S8ADDQ instruction.
 */
#define MASK6           (0x3F << 3)
#else
#ifdef CRYPT_MOVZBL
/*
 * Masking whole bytes allows the compiler to use Move with Zero Extension
 * instructions (where supported), like MOVZBL (MOVZX in Intel's syntax) on
 * x86s, or EXTBL on DEC Alphas. It might only be reasonable to disable this
 * if such instructions exist, but are slower than masks/shifts (as they are
 * on Pentiums).
 */
#define MASK6           ((0x3F << SIZEFIX) | 0xFF)
#else
/*
 * Forces using plain shifts/masks, sometimes it's the only choice. Note that
 * you don't have to define CRYPT_MASK if this is the case -- walking pointer
 * method might be faster.
 */
#define MASK6           (0x3F << SIZEFIX)
#endif
#endif
#endif
#define MASK16          DO_SIZEFIX(0x1E1E1E1E)

/*
 * The entire loop is unrolled by default. The only reason for unrolling parts
 * of the loop might be having a small I-cache.
 */
#ifndef CRYPT_UNROLL
#define CRYPT_UNROLL 8
#endif

#if CRYPT_UNROLL == 8
/*
 * Unrolling the entire loop allows using fixed KS element addresses, we save
 * a register for better use.
 */
#define B_KS(i)         ((sbpb24 *)(((unsigned char (*)[16]) KS) + (i)))
#else
/*
 * Otherwise a walking pointer is used.
 */
#define B_KS(i)         ((sbpb24 *)(((unsigned char (*)[16]) kp) + (i)))
#endif

#define KEYSIZE         (16 * 8)

/*
 * This used to be rather slow and frequently used function - AEM.
 * It is not frequently used any more - SD.
 */

#define TF_TO_SIXBIT(tf) \
	(sbpb24)((tf & 077L) | \
		((tf & 07700L) << 2) | \
		((tf & 0770000L) << 4) | \
		((tf & 077000000L) << 6))

static obpb1 crypt_block[72];   /* 72 is next multiple of 8 bytes after 66 */

/*
 * Data type to store binary encrypted passwords, only 64 of 128 bits used.
 */
typedef sbpb24 crypt_bin[4 / SIZEDIV];
#define crypt_binsize sizeof(crypt_bin)

/*
 * crypt_binary() and XForm1()/XForm2() result buffers, respectively.
 */
crypt_bin crypt_cbin;
EXTERN crypt_bin crypt_cout;

#ifndef CRYPT_MIXED
/*
 * A shift to avoid L1 cache bank conflicts (and allow pairing) on Pentiums.
 * Thanks to Roman Rusakov for this idea - SD.
 */
#define BANKSHIFT       1
#else
#define BANKSHIFT       0
#endif

#ifndef CRYPT_128K
EXTERN
#endif
sbpb24 SALL[(1024 + BANKSHIFT) / SIZEDIV][1];

EXTERN sbpb24 KS[32 / SIZEDIV];

#ifndef CRYPT_128K
EXTERN sbpb24 SCUR[(1024 + BANKSHIFT) / SIZEDIV][1];
#endif

/*
 * This stuff looks ugly, I know, but it seems to be the simplest way to make
 * the compiler put variables in the correct order, and not spread them all
 * over the data segment. SDEF is defined to either SALL or SCUR when compiling
 * XForm1 or XForm2, respectively - SD.
 */

#ifndef CRYPT_64BIT
#ifndef CRYPT_MIXED
#define S0L SDEF[0 * 64]
#define S1L SDEF[1 * 64]
#define S2L SDEF[2 * 64]
#define S3L SDEF[3 * 64]
#define S4L SDEF[4 * 64]
#define S5L SDEF[5 * 64]
#define S6L SDEF[6 * 64]
#define S7L SDEF[7 * 64]
#define S0H SDEF[8 * 64 + BANKSHIFT]
#define S1H SDEF[9 * 64 + BANKSHIFT]
#define S2H SDEF[10 * 64 + BANKSHIFT]
#define S3H SDEF[11 * 64 + BANKSHIFT]
#define S4H SDEF[12 * 64 + BANKSHIFT]
#define S5H SDEF[13 * 64 + BANKSHIFT]
#define S6H SDEF[14 * 64 + BANKSHIFT]
#define S7H SDEF[15 * 64 + BANKSHIFT]
#else
#define S0L SDEF[0 * 64]
#define S1L SDEF[2 * 64]
#define S2L SDEF[4 * 64]
#define S3L SDEF[6 * 64]
#define S4L SDEF[8 * 64]
#define S5L SDEF[10 * 64]
#define S6L SDEF[12 * 64]
#define S7L SDEF[14 * 64]
#define S0H SDEF[0 * 64 + 1]
#define S1H SDEF[2 * 64 + 1]
#define S2H SDEF[4 * 64 + 1]
#define S3H SDEF[6 * 64 + 1]
#define S4H SDEF[8 * 64 + 1]
#define S5H SDEF[10 * 64 + 1]
#define S6H SDEF[12 * 64 + 1]
#define S7H SDEF[14 * 64 + 1]
#endif
#else
#define S0 SDEF[0 * 64]
#define S1 SDEF[1 * 64]
#define S2 SDEF[2 * 64]
#define S3 SDEF[3 * 64]
#define S4 SDEF[4 * 64]
#define S5 SDEF[5 * 64]
#define S6 SDEF[6 * 64]
#define S7 SDEF[7 * 64]
#endif

#ifndef CRYPT_128K
#ifndef CRYPT_64BIT
#ifndef CRYPT_MIXED
#define S0LC SCUR[0 * 64]
#define S1LC SCUR[1 * 64]
#define S2LC SCUR[2 * 64]
#define S3LC SCUR[3 * 64]
#define S4LC SCUR[4 * 64]
#define S5LC SCUR[5 * 64]
#define S6LC SCUR[6 * 64]
#define S7LC SCUR[7 * 64]
#define S0HC SCUR[8 * 64 + BANKSHIFT]
#define S1HC SCUR[9 * 64 + BANKSHIFT]
#define S2HC SCUR[10 * 64 + BANKSHIFT]
#define S3HC SCUR[11 * 64 + BANKSHIFT]
#define S4HC SCUR[12 * 64 + BANKSHIFT]
#define S5HC SCUR[13 * 64 + BANKSHIFT]
#define S6HC SCUR[14 * 64 + BANKSHIFT]
#define S7HC SCUR[15 * 64 + BANKSHIFT]
#else
#define S0LC SCUR[0 * 64]
#define S1LC SCUR[2 * 64]
#define S2LC SCUR[4 * 64]
#define S3LC SCUR[6 * 64]
#define S4LC SCUR[8 * 64]
#define S5LC SCUR[10 * 64]
#define S6LC SCUR[12 * 64]
#define S7LC SCUR[14 * 64]
#define S0HC SCUR[0 * 64 + 1]
#define S1HC SCUR[2 * 64 + 1]
#define S2HC SCUR[4 * 64 + 1]
#define S3HC SCUR[6 * 64 + 1]
#define S4HC SCUR[8 * 64 + 1]
#define S5HC SCUR[10 * 64 + 1]
#define S6HC SCUR[12 * 64 + 1]
#define S7HC SCUR[14 * 64 + 1]
#endif
#else
#define S0C SCUR[0 * 64]
#define S1C SCUR[1 * 64]
#define S2C SCUR[2 * 64]
#define S3C SCUR[3 * 64]
#define S4C SCUR[4 * 64]
#define S5C SCUR[5 * 64]
#define S6C SCUR[6 * 64]
#define S7C SCUR[7 * 64]
#endif
#else
EXTERN sbpb24 SWALL[(32768 + BANKSHIFT) / SIZEDIV][1];
EXTERN sbpb24 SWCUR[(32768 + BANKSHIFT) / SIZEDIV][1];
#ifndef CRYPT_64BIT
#ifndef CRYPT_MIXED
#define SW0L SWDEF[0 * 4096]
#define SW1L SWDEF[1 * 4096]
#define SW2L SWDEF[2 * 4096]
#define SW3L SWDEF[3 * 4096]
#define SW0H SWDEF[4 * 4096 + BANKSHIFT]
#define SW1H SWDEF[5 * 4096 + BANKSHIFT]
#define SW2H SWDEF[6 * 4096 + BANKSHIFT]
#define SW3H SWDEF[7 * 4096 + BANKSHIFT]
#define SW0LC SWCUR[0 * 4096]
#define SW1LC SWCUR[1 * 4096]
#define SW2LC SWCUR[2 * 4096]
#define SW3LC SWCUR[3 * 4096]
#define SW0HC SWCUR[4 * 4096 + BANKSHIFT]
#define SW1HC SWCUR[5 * 4096 + BANKSHIFT]
#define SW2HC SWCUR[6 * 4096 + BANKSHIFT]
#define SW3HC SWCUR[7 * 4096 + BANKSHIFT]
#else
#define SW0L SWDEF[0 * 4096]
#define SW1L SWDEF[2 * 4096]
#define SW2L SWDEF[4 * 4096]
#define SW3L SWDEF[6 * 4096]
#define SW0H SWDEF[0 * 4096 + 1]
#define SW1H SWDEF[2 * 4096 + 1]
#define SW2H SWDEF[4 * 4096 + 1]
#define SW3H SWDEF[6 * 4096 + 1]
#define SW0LC SWCUR[0 * 4096]
#define SW1LC SWCUR[2 * 4096]
#define SW2LC SWCUR[4 * 4096]
#define SW3LC SWCUR[6 * 4096]
#define SW0HC SWCUR[0 * 4096 + 1]
#define SW1HC SWCUR[2 * 4096 + 1]
#define SW2HC SWCUR[4 * 4096 + 1]
#define SW3HC SWCUR[6 * 4096 + 1]
#endif
#else
#define SW0 SWDEF[0 * 4096]
#define SW1 SWDEF[1 * 4096]
#define SW2 SWDEF[2 * 4096]
#define SW3 SWDEF[3 * 4096]
#define SW0C SWCUR[0 * 4096]
#define SW1C SWCUR[1 * 4096]
#define SW2C SWCUR[2 * 4096]
#define SW3C SWCUR[3 * 4096]
#endif
#endif

/* Another great Roman Rusakov's idea */
EXTERN sbpb24 BigKS[8][128][32 / SIZEDIV];
char lastpw[16];

/*
 * fsetkey() is no longer time critical, so I reduced its size a little - SD.
 */

void
fsetkey ()
{
    static unsb KeyToKS[] =
    {
	9, 50, 33, 59, 48, 16, 32, 56, 1, 8, 18, 41, 2, 34, 25, 24,
	43, 57, 58, 0, 35, 26, 17, 40, 21, 27, 38, 53, 36, 3, 46, 29,
	4, 52, 22, 28, 60, 20, 37, 62, 14, 19, 44, 13, 12, 61, 54, 30,
	1, 42, 25, 51, 40, 8, 24, 48, 58, 0, 10, 33, 59, 26, 17, 16,
	35, 49, 50, 57, 56, 18, 9, 32, 13, 19, 30, 45, 28, 62, 38, 21,
	27, 44, 14, 20, 52, 12, 29, 54, 6, 11, 36, 5, 4, 53, 46, 22,
	50, 26, 9, 35, 24, 57, 8, 32, 42, 49, 59, 17, 43, 10, 1, 0,
	48, 33, 34, 41, 40, 2, 58, 16, 60, 3, 14, 29, 12, 46, 22, 5,
	11, 28, 61, 4, 36, 27, 13, 38, 53, 62, 20, 52, 19, 37, 30, 6,
	34, 10, 58, 48, 8, 41, 57, 16, 26, 33, 43, 1, 56, 59, 50, 49,
	32, 17, 18, 25, 24, 51, 42, 0, 44, 54, 61, 13, 27, 30, 6, 52,
	62, 12, 45, 19, 20, 11, 60, 22, 37, 46, 4, 36, 3, 21, 14, 53,
	18, 59, 42, 32, 57, 25, 41, 0, 10, 17, 56, 50, 40, 43, 34, 33,
	16, 1, 2, 9, 8, 35, 26, 49, 28, 38, 45, 60, 11, 14, 53, 36,
	46, 27, 29, 3, 4, 62, 44, 6, 21, 30, 19, 20, 54, 5, 61, 37,
	2, 43, 26, 16, 41, 9, 25, 49, 59, 1, 40, 34, 24, 56, 18, 17,
	0, 50, 51, 58, 57, 48, 10, 33, 12, 22, 29, 44, 62, 61, 37, 20,
	30, 11, 13, 54, 19, 46, 28, 53, 5, 14, 3, 4, 38, 52, 45, 21,
	51, 56, 10, 0, 25, 58, 9, 33, 43, 50, 24, 18, 8, 40, 2, 1,
	49, 34, 35, 42, 41, 32, 59, 17, 27, 6, 13, 28, 46, 45, 21, 4,
	14, 62, 60, 38, 3, 30, 12, 37, 52, 61, 54, 19, 22, 36, 29, 5,
	35, 40, 59, 49, 9, 42, 58, 17, 56, 34, 8, 2, 57, 24, 51, 50,
	33, 18, 48, 26, 25, 16, 43, 1, 11, 53, 60, 12, 30, 29, 5, 19,
	61, 46, 44, 22, 54, 14, 27, 21, 36, 45, 38, 3, 6, 20, 13, 52,
	56, 32, 51, 41, 1, 34, 50, 9, 48, 26, 0, 59, 49, 16, 43, 42,
	25, 10, 40, 18, 17, 8, 35, 58, 3, 45, 52, 4, 22, 21, 60, 11,
	53, 38, 36, 14, 46, 6, 19, 13, 28, 37, 30, 62, 61, 12, 5, 44,
	40, 16, 35, 25, 50, 18, 34, 58, 32, 10, 49, 43, 33, 0, 56, 26,
	9, 59, 24, 2, 1, 57, 48, 42, 54, 29, 36, 19, 6, 5, 44, 62,
	37, 22, 20, 61, 30, 53, 3, 60, 12, 21, 14, 46, 45, 27, 52, 28,
	24, 0, 48, 9, 34, 2, 18, 42, 16, 59, 33, 56, 17, 49, 40, 10,
	58, 43, 8, 51, 50, 41, 32, 26, 38, 13, 20, 3, 53, 52, 28, 46,
	21, 6, 4, 45, 14, 37, 54, 44, 27, 5, 61, 30, 29, 11, 36, 12,
	8, 49, 32, 58, 18, 51, 2, 26, 0, 43, 17, 40, 1, 33, 24, 59,
	42, 56, 57, 35, 34, 25, 16, 10, 22, 60, 4, 54, 37, 36, 12, 30,
	5, 53, 19, 29, 61, 21, 38, 28, 11, 52, 45, 14, 13, 62, 20, 27,
	57, 33, 16, 42, 2, 35, 51, 10, 49, 56, 1, 24, 50, 17, 8, 43,
	26, 40, 41, 48, 18, 9, 0, 59, 6, 44, 19, 38, 21, 20, 27, 14,
	52, 37, 3, 13, 45, 5, 22, 12, 62, 36, 29, 61, 60, 46, 4, 11,
	41, 17, 0, 26, 51, 48, 35, 59, 33, 40, 50, 8, 34, 1, 57, 56,
	10, 24, 25, 32, 2, 58, 49, 43, 53, 28, 3, 22, 5, 4, 11, 61,
	36, 21, 54, 60, 29, 52, 6, 27, 46, 20, 13, 45, 44, 30, 19, 62,
	25, 1, 49, 10, 35, 32, 48, 43, 17, 24, 34, 57, 18, 50, 41, 40,
	59, 8, 9, 16, 51, 42, 33, 56, 37, 12, 54, 6, 52, 19, 62, 45,
	20, 5, 38, 44, 13, 36, 53, 11, 30, 4, 60, 29, 28, 14, 3, 46,
	17, 58, 41, 2, 56, 24, 40, 35, 9, 16, 26, 49, 10, 42, 33, 32,
	51, 0, 1, 8, 43, 34, 25, 48, 29, 4, 46, 61, 44, 11, 54, 37,
	12, 60, 30, 36, 5, 28, 45, 3, 22, 27, 52, 21, 20, 6, 62, 38
    };

    reg int i, j, k;
    reg unsb *l = KeyToKS;
    reg unsigned long r;

    for (i = 0; i < 32; i++)
    {
        for (r = j = 0; j < 32; j += 8)
        for (k = 0; k < 6; k++)
            r |= (unsigned long) crypt_block[*l++] << (j + k);

#ifndef CRYPT_64BIT
	KS[i] = DO_SIZEFIX (r);
#else
        if (i & 1) KS[i >> 1] |= (sbpb24)DO_SIZEFIX(r) << 32;
        else KS[i >> 1] = DO_SIZEFIX(r);
#endif
    }
}

#ifdef CRYPT_ASM

extern void xorkey1();
extern void xorkey2();

#else

/*
 * These are made inline functions (not macros) to make the compiler calculate
 * addresses outside of the loops - SD.
 */

#define xor1(ofs) KS[ofs] ^= val[ofs];
#define xor1_4(ofs) \
    xor1(ofs + 0); xor1(ofs + 1); xor1(ofs + 2); xor1(ofs + 3);
#define xor1_16(ofs) \
    xor1_4(ofs + 0); xor1_4(ofs + 4); xor1_4(ofs + 8); xor1_4(ofs + 12);

inline void xorkey1(sbpb24 *val) {
    xor1_16(0);
#ifndef CRYPT_64BIT
    xor1_16(16);
#endif
}

#define xor2(ofs) KS[ofs] ^= val1[ofs] ^ val2[ofs];
#define xor2_4(ofs) \
    xor2(ofs + 0); xor2(ofs + 1); xor2(ofs + 2); xor2(ofs + 3);
#define xor2_16(ofs) \
    xor2_4(ofs + 0); xor2_4(ofs + 4); xor2_4(ofs + 8); xor2_4(ofs + 12);

inline void xorkey2(sbpb24 *val1, sbpb24 *val2) {
    xor2_16(0);
#ifndef CRYPT_64BIT
    xor2_16(16);
#endif
}

#endif

/*
 * The new crypt_setkey() idea is originally by Roman Rusakov. I extended
 * the original idea to detect whether it's faster to calculate from scratch
 * or modify the previous key. This code assumes that xorkey1() is 1.5 times
 * faster than xorkey2() - SD.
 */

void
crypt_setkey(pw)
    char *pw;
{
    reg int i, j, k, l;

    for (k = i = 0; (j = pw[i]) && (l = lastpw[i]); i++)
    if (j != l) k += 3;

    if (j) {
        while (i < 8 && pw[i]) {
          i++; k += 2;
        }
        j = i;
    } else {
        j = i; while (lastpw[i++]) k += 2;
    }

    if (k < (j << 1)) {
        for (i = 0; (k = pw[i]) && (l = lastpw[i]); i++)
        if (k != l) xorkey2(BigKS[i][l], BigKS[i][k]);

        if (k) do
            xorkey1(BigKS[i][k]);
        while ((k = pw[++i]) && i < 8);
        else if (l = lastpw[i]) do
            xorkey1(BigKS[i][l]);
        while (l = lastpw[++i]);
    } else {
        memcpy(KS, BigKS[0][k = pw[0]], KEYSIZE);

        if (k)
        for (i = 1; (k = pw[i]) && i < 8; i++) xorkey1(BigKS[i][k]);
    }

    memcpy(lastpw, pw, 8);
    lastpw[8] = 0;
}

#define SDEF SALL
#define SWDEF SWALL

EXTERN sbpb24 saltvalue;

#ifdef CRYPT_ASM

extern void XForm1 ();
extern void XForm2 ();

#else

#ifndef CRYPT_128K

/*
 * Walking pointer is replaced with masks/shifts when using 12-bit indices,
 * not to get improperly aligned memory accesses on many architectures - SD.
 */

#ifdef CRYPT_64BIT
#define Db sdata.b[0]
#endif

#ifndef CRYPT_LENDIAN    	/* Icarus Sparry, Bath - mod AEM */
#define STEP --
#define START &sdata.c[7]
#define Dl sdata.b[1]
#define Dh sdata.b[0]
#else
#define STEP ++
#define START &sdata.c[0]
#define Dl sdata.b[0]
#define Dh sdata.b[1]
#endif

#endif

/*
 * The code below is heavily optimized, looking at the assembly output for
 * several architectures, enjoy the speed - SD.
 */

#ifndef CRYPT_64BIT
#ifndef CRYPT_128K
#ifdef CRYPT_MASK
/* 32 bit, 4K, masks */
#define LXForm0(L, H, B) \
    L ^= INDIRECT (S0L, Dl & MASK6); \
    H ^= INDIRECT (S0H, Dl & MASK6); \
    L ^= INDIRECT (S1L, (Dl >>= 8) & MASK6); \
    H ^= INDIRECT (S1H, Dl & MASK6); \
    L ^= INDIRECT (S2L, (Dl >>= 8) & MASK6); \
    H ^= INDIRECT (S2H, Dl & MASK6); \
    L ^= INDIRECT (S3L, (Dl >>= 8) & MASK6); \
    H ^= INDIRECT (S3H, Dl & MASK6); \
    L ^= INDIRECT (S4L, Dh & MASK6); \
    H ^= INDIRECT (S4H, Dh & MASK6); \
    L ^= INDIRECT (S5L, (Dh >>= 8) & MASK6); \
    H ^= INDIRECT (S5H, Dh & MASK6); \
    L ^= INDIRECT (S6L, (Dh >>= 8) & MASK6); \
    H ^= INDIRECT (S6H, Dh & MASK6); \
    L ^= INDIRECT (S7L, (Dh >>= 8) & MASK6); \
    H ^= INDIRECT (S7H, Dh & MASK6)
#else
/* 32 bit, 4K, walking pointer */
#define LXForm0(L, H, B) \
    dp = START; \
    L ^= INDIRECT (S0L, *dp); \
    H ^= INDIRECT (S0H, *dp STEP); \
    L ^= INDIRECT (S1L, *dp); \
    H ^= INDIRECT (S1H, *dp STEP); \
    L ^= INDIRECT (S2L, *dp); \
    H ^= INDIRECT (S2H, *dp STEP); \
    L ^= INDIRECT (S3L, *dp); \
    H ^= INDIRECT (S3H, *dp STEP); \
    L ^= INDIRECT (S4L, *dp); \
    H ^= INDIRECT (S4H, *dp STEP); \
    L ^= INDIRECT (S5L, *dp); \
    H ^= INDIRECT (S5H, *dp STEP); \
    L ^= INDIRECT (S6L, *dp); \
    H ^= INDIRECT (S6H, *dp STEP); \
    L ^= INDIRECT (S7L, *dp); \
    H ^= INDIRECT (S7H, *dp STEP)
#endif
#else
/* 32 bit, 128K, masks */
#define LXForm0(L, H, B) \
    L ^= INDIRECT (SW0L, Dl & 0xFFFF); \
    H ^= INDIRECT (SW0H, Dl & 0xFFFF); \
    L ^= INDIRECT (SW1L, Dl >>= 16); \
    H ^= INDIRECT (SW1H, Dl); \
    L ^= INDIRECT (SW2L, Dh & 0xFFFF); \
    H ^= INDIRECT (SW2H, Dh & 0xFFFF); \
    L ^= INDIRECT (SW3L, Dh >>= 16); \
    H ^= INDIRECT (SW3H, Dh)
#endif
#else
#ifndef CRYPT_128K
/*
 * An extra temporary register is used for better instruction scheduling. We
 * still usually get a pipeline stall in the last two instructions though, I
 * couldn't find a way to fix this without using assembler - SD.
 */
#ifdef CRYPT_MASK
#ifdef CRYPT_SCALE
/* 64 bit, 4K, masks, scale */ 
#define LXForm0(L, H, B) \
    T = INDIRECT (S0, Db & MASK6); \
    B ^= INDIRECT (S1, (Db >> 8) & MASK6); \
    T ^= INDIRECT (S2, (Db >> 16) & MASK6); \
    B ^= INDIRECT (S3, (Db >> 24) & MASK6); \
    T ^= INDIRECT (S4, (Db >> 32) & MASK6); \
    B ^= INDIRECT (S5, (Db >> 40) & MASK6); \
    T ^= INDIRECT (S6, (Db >> 48) & MASK6); \
    B ^= T ^ INDIRECT (S7, Db >> 56)
#else
/*
 * This code assumes that the indices are already shifted left by 2, and still
 * should be shifted left by 1 more bit. Weird shift counts allow doing this
 * with only one extra shift, in the first line. By shifting the mask right
 * (which is done at compile time), I moved the extra shift out of the mask,
 * which allows doing this with special addressing modes, where possible - SD.
 */
/* 64 bit, 4K, masks, no scale */
#define LXForm0(L, H, B) \
    T = INDIRECT (S0, (Db & (MASK6 >> 1)) << 1); \
    B ^= INDIRECT (S1, (Db >> 7) & MASK6); \
    T ^= INDIRECT (S2, (Db >> 15) & MASK6); \
    B ^= INDIRECT (S3, (Db >> 23) & MASK6); \
    T ^= INDIRECT (S4, (Db >> 31) & MASK6); \
    B ^= INDIRECT (S5, (Db >> 39) & MASK6); \
    T ^= INDIRECT (S6, (Db >> 47) & MASK6); \
    B ^= T ^ INDIRECT (S7, (Db >> 55) & MASK6)
#endif
#else
/* 64 bit, 4K, walking pointer, scale */
#define LXForm0(L, H, B) \
    dp = START; \
    T = INDIRECT (S0, *dp STEP); \
    B ^= INDIRECT (S1, *dp STEP); \
    T ^= INDIRECT (S2, *dp STEP); \
    B ^= INDIRECT (S3, *dp STEP); \
    T ^= INDIRECT (S4, *dp STEP); \
    B ^= INDIRECT (S5, *dp STEP); \
    T ^= INDIRECT (S6, *dp STEP); \
    B ^= T ^ INDIRECT (S7, *dp STEP)
#endif
#else
/* 64 bit, 128K, masks, no scale */
#define LXForm0(L, H, B) \
    B ^= INDIRECT (SW0, Db & 0xFFFF); \
    B ^= INDIRECT (SW1, (Db >>= 16) & 0xFFFF); \
    B ^= INDIRECT (SW2, (Db >>= 16) & 0xFFFF); \
    B ^= INDIRECT (SW3, Db >> 16)
#endif
#endif

#ifndef CRYPT_64BIT
#define LXForm1(MYKS) \
    k = (Rl ^ Rh) & saltvalue; \
    Dl = k ^ Rl ^ MYKS[0]; \
    Dh = k ^ Rh ^ MYKS[1]; \
    LXForm0 (Ll, Lh, L); \
    k = (Ll ^ Lh) & saltvalue; \
    Dl = k ^ Ll ^ MYKS[2]; \
    Dh = k ^ Lh ^ MYKS[3]; \
    LXForm0 (Rl, Rh, R);
#else
#define LXForm1(MYKS) \
    k = (R ^ (R >> 32)) & saltvalue; \
    k |= k << 32; \
    Db = k ^ R ^ MYKS[0]; \
    LXForm0 (Ll, Lh, L); \
    k = (L ^ (L >> 32)) & saltvalue; \
    k |= k << 32; \
    Db = k ^ L ^ MYKS[1]; \
    LXForm0 (Rl, Rh, R);
#endif

#ifndef CRYPT_64BIT
#define LXForm2(MYKS) \
    Dl = Rl ^ MYKS[0]; \
    Dh = Rh ^ MYKS[1]; \
    LXForm0 (Ll, Lh, L); \
    Dl = Ll ^ MYKS[2]; \
    Dh = Lh ^ MYKS[3]; \
    LXForm0 (Rl, Rh, R);
#else
#define LXForm2(MYKS) \
    Db = R ^ MYKS[0]; \
    LXForm0 (Ll, Lh, L); \
    Db = L ^ MYKS[1]; \
    LXForm0 (Rl, Rh, R);
#endif

#if CRYPT_UNROLL == 8
#define UnrollXForm(ThisOne) \
    ThisOne(B_KS(0)); ThisOne(B_KS(1)); ThisOne(B_KS(2)); ThisOne(B_KS(3)); \
    ThisOne(B_KS(4)); ThisOne(B_KS(5)); ThisOne(B_KS(6)); ThisOne(B_KS(7));
#endif
#if CRYPT_UNROLL == 4
#define UnrollXForm(ThisOne) \
    for (kp = KS; kp < KS + (32 / SIZEDIV); kp += (16 / SIZEDIV)) { \
        ThisOne(B_KS(0)); ThisOne(B_KS(1)); ThisOne(B_KS(2)); ThisOne(B_KS(3)); \
    }
#endif
#if CRYPT_UNROLL == 2
#define UnrollXForm(ThisOne) \
    for (kp = KS; kp < KS + (32 / SIZEDIV); kp += (8 / SIZEDIV)) { \
        ThisOne(B_KS(0)); ThisOne(B_KS(1)); \
    }
#endif
#ifndef UnrollXForm
#define UnrollXForm(ThisOne) \
    for (kp = KS; kp < KS + (32 / SIZEDIV); kp += (4 / SIZEDIV)) { \
        ThisOne(kp); \
    }
#endif

void XForm1 () {
#ifndef CRYPT_128K
    union SDATA sdata;
#ifndef CRYPT_MASK
    reg sbpb6 *dp;
#endif
#else
#ifndef CRYPT_64BIT
    reg sbpb24 Dl, Dh;
#else
    reg sbpb24 Db;
#endif
#endif
#ifndef CRYPT_64BIT
    reg sbpb24 Rl, Rh, Ll, Lh;
#else
    reg sbpb24 R, L;
#ifndef CRYPT_128K
    reg sbpb24 T;
#endif
#endif
    int loopcount;
    sbpb24 k;
#if CRYPT_UNROLL != 8
    sbpb24 *kp;
#endif

#ifndef CRYPT_64BIT
    Ll = Lh = Rl = Rh = 0;
#else
    L = R = 0;
#endif

    loopcount = 25;
    do {
        UnrollXForm(LXForm1);

#ifndef CRYPT_64BIT
	Ll ^= Rl;
	Lh ^= Rh;
	Rl ^= Ll;
	Rh ^= Lh;
	Ll ^= Rl;
	Lh ^= Rh;
#else
        L ^= R;
        R ^= L;
        L ^= R;
#endif
    } while (--loopcount);

#ifndef CRYPT_64BIT
    crypt_cout[0] = Ll & MASK16;
    crypt_cout[1] = Lh & MASK16;
    crypt_cout[2] = Rl & MASK16;
    crypt_cout[3] = Rh & MASK16;
#else
    crypt_cout[0] = L & (MASK16 | ((sbpb24)MASK16 << 32));
    crypt_cout[1] = R & (MASK16 | ((sbpb24)MASK16 << 32));
#endif
}

#undef SDEF
#undef SWDEF
#define SDEF SCUR
#define SWDEF SWCUR

void XForm2 () {
#ifndef CRYPT_128K
    union SDATA sdata;
#ifndef CRYPT_MASK
    reg sbpb6 *dp;
#endif
#else
#ifndef CRYPT_64BIT
    reg sbpb24 Dl, Dh;
#else
    reg sbpb24 Db;
#endif
#endif
#ifndef CRYPT_64BIT
    reg sbpb24 Rl, Rh, Ll, Lh;
#else
    reg sbpb24 R, L;
#ifndef CRYPT_128K
    reg sbpb24 T;
#endif
#endif
    int loopcount;
#if CRYPT_UNROLL != 8
    sbpb24 *kp;
#endif

#ifndef CRYPT_64BIT
    Ll = Lh = Rl = Rh = 0;
#else
    L = R = 0;
#endif

    loopcount = 25;
    do {
        UnrollXForm(LXForm2);

#ifndef CRYPT_64BIT
	Ll ^= Rl;
	Lh ^= Rh;
	Rl ^= Ll;
	Rh ^= Lh;
	Ll ^= Rl;
	Lh ^= Rh;
#else
        L ^= R;
        R ^= L;
        L ^= R;
#endif
    } while (--loopcount);

#ifndef CRYPT_64BIT
    crypt_cout[0] = Ll & MASK16;
    crypt_cout[1] = Lh & MASK16;
    crypt_cout[2] = Rl & MASK16;
    crypt_cout[3] = Rh & MASK16;
#else
    crypt_cout[0] = L & (MASK16 | ((sbpb24)MASK16 << 32));
    crypt_cout[1] = R & (MASK16 | ((sbpb24)MASK16 << 32));
#endif
}

#undef SDEF
#undef SWDEF
#define SDEF SALL
#define SWDEF SWALL

#endif

crypt_bin *
crypt_check (_saltvalue)
    sbpb24 _saltvalue;
{
    saltvalue = _saltvalue;
    XForm1 ();
    return &crypt_cout;
}

unsigned short
crypt_getsalt (salt)
    char *salt;
{
    static unsigned char salt0[] =
    {
        18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
        32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
        48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 5, 6, 7, 8, 9, 10, 11, 12,
        13, 14, 15, 16, 17,

        18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
        32, 33, 34, 35, 36, 37, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41,
        42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57,
        58, 59, 60, 61, 62, 63, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11,
        12, 13, 14, 15, 16, 17,

        18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
        32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
        48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17,

        18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
        32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
        48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
        0, 1, 2, 3, 4
    };

    return salt0[salt[0]] | (unsigned short)salt0[salt[1]] << 6;
}

unsigned short
crypt_fixsalt (salt)
    unsigned short salt;
{
#ifdef CRYPT_128K
    return salt << SIZEFIX;
#else
    salt <<= 2;
#if SIZEFIX == 0
    return ((salt & 0x00FF) >> 2) | (salt & 0xFF00);
#else
    return (salt & 0x00FF) | ((salt & 0xFF00) << 2);
#endif
#endif
}

void
init_salt (src, dst)
    sbpb24 *src, *dst;
{
#ifndef CRYPT_64BIT
#ifndef CRYPT_MIXED
#ifndef CRYPT_128K
    reg int i = 0, j = 8 * 64 + BANKSHIFT;
#else
    reg int i = 0, j = 4 * 4096 + BANKSHIFT;
#endif
#else
    reg int i = 0, j = 1;
#endif
    reg sbpb24 spare24, low24, high24;

    do {
        dst[i] = (spare24 = ((low24 = src[i]) ^ (high24 = src[j])) & saltvalue) ^ low24;
        dst[j] = spare24 ^ high24;
#ifndef CRYPT_MIXED
        i++; j++;
#ifndef CRYPT_128K
    } while (i < 64);
#else
    } while (i < 4096);
#endif
#else
    i += 2; j += 2;
#ifndef CRYPT_128K
    } while (i < 2 * 64);
#else
    } while (i < 2 * 4096);
#endif
#endif
#else
    reg int i = 0;
    reg sbpb24 spare;

    do {
        spare = (src[i] ^ (src[i] >> 32)) & saltvalue;
        dst[i] = src[i] ^ (spare | (spare << 32));
        i++;
#ifndef CRYPT_128K
    } while (i < 64);
#else
    } while (i < 4096);
#endif
#endif
}

/*
 * crypt_setsalt() sets a salt for use with XForm2().
 */

void
crypt_setsalt (salt)
    unsigned short salt;
{
    saltvalue = TF_TO_SIXBIT (salt);
    saltvalue = DO_SIZEFIX (saltvalue);

#ifndef CRYPT_64BIT
#ifndef CRYPT_128K
    init_salt (S0L, S0LC);
    init_salt (S1L, S1LC);
    init_salt (S2L, S2LC);
    init_salt (S3L, S3LC);
    init_salt (S4L, S4LC);
    init_salt (S5L, S5LC);
    init_salt (S6L, S6LC);
    init_salt (S7L, S7LC);
#else
    init_salt (SW0L, SW0LC);
    init_salt (SW1L, SW1LC);
    init_salt (SW2L, SW2LC);
    init_salt (SW3L, SW3LC);
#endif
#else
#ifndef CRYPT_128K
    init_salt (S0, S0C);
    init_salt (S1, S1C);
    init_salt (S2, S2C);
    init_salt (S3, S3C);
    init_salt (S4, S4C);
    init_salt (S5, S5C);
    init_salt (S6, S6C);
    init_salt (S7, S7C);
#else
    init_salt (SW0, SW0C);
    init_salt (SW1, SW1C);
    init_salt (SW2, SW2C);
    init_salt (SW3, SW3C);
#endif
#endif
}

/*
 * This function does the reverse of what UnXForm used to do, converts an
 * encrypted password from ASCII to binary, to be compared to the XForm's
 * result - SD.
 */

crypt_bin *
crypt_binary (passwd, usesalt)
    char *passwd;
    int usesalt;
{
    reg int i, j, k;
    reg char c;
    reg sbpb24 Rl, Rh, Ll, Lh;
    reg sbpb24 mask1, mask2;
    reg obpb1 *ptr;

    static obpb1 final[] =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,
        2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 0,
        0, 0, 0, 0, 0, 12, 13, 14, 15, 16, 17, 18,
        19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
        31, 32, 33, 34, 35, 36, 37, 0, 0, 0, 0, 0,
        0, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
        49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60,
        61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72,
        73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84,
        85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96,
        97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108,
        109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
        121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132,
        133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144,
        145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156,
        157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168,
        169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180,
        181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192,
        193, 194, 195, 196
    };

    static sbpb24 masks[] =
    {
        0x400, 0x200, 0x100, 0x080, 0x010, 0x008, 0x004, 0x002
    };

    for (i = 0; i < 11; i++)
    {
        c = final[passwd[i + 2]];
        k = 0x20;

        for (j = 0; j < 6; j++)
        {
            crypt_block[i * 6 + j] = ((c & k) != 0) & 1;
            k >>= 1;
        }
    }

    Ll = Lh = Rl = Rh = 0;

    ptr = crypt_block;
    for (i = 0; i < 8; i++)
    {
        mask2 = (mask1 = masks[i]) << 12;
        if (*ptr++) Rl |= mask1;
        if (*ptr++) Ll |= mask1;
        if (*ptr++) Rl |= mask2;
        if (*ptr++) Ll |= mask2;
        if (*ptr++) Rh |= mask1;
        if (*ptr++) Lh |= mask1;
        if (*ptr++) Rh |= mask2;
        if (*ptr++) Lh |= mask2;
    }

    if (usesalt) {
        mask1 = (Ll ^ Lh) & (mask2 = crypt_getsalt(passwd));
        mask2 &= Rl ^ Rh;
        Ll ^= mask1;
        Lh ^= mask1;
        Rl ^= mask2;
        Rh ^= mask2;
    }

    Ll = TF_TO_SIXBIT (Ll);
    Lh = TF_TO_SIXBIT (Lh);
    Rl = TF_TO_SIXBIT (Rl);
    Rh = TF_TO_SIXBIT (Rh);

#ifndef CRYPT_64BIT
    crypt_cbin[0] = DO_SIZEFIX (Ll);
    crypt_cbin[1] = DO_SIZEFIX (Lh);
    crypt_cbin[2] = DO_SIZEFIX (Rl);
    crypt_cbin[3] = DO_SIZEFIX (Rh);
#else
    crypt_cbin[0] = DO_SIZEFIX (Ll) | ((sbpb24)DO_SIZEFIX (Lh) << 32);
    crypt_cbin[1] = DO_SIZEFIX (Rl) | ((sbpb24)DO_SIZEFIX (Rh) << 32);
#endif

    return &crypt_cbin;
}

fbpb4
lookupS (tableno, t6bits)
    unsl tableno;
    sbpb6R t6bits;
{
    static fbpb4R S[8][64] =
    {
	{14, 4, 13, 1, 2, 15, 11, 8, 3, 10, 6, 12, 5, 9, 0, 7,
	0, 15, 7, 4, 14, 2, 13, 1, 10, 6, 12, 11, 9, 5, 3, 8,
	4, 1, 14, 8, 13, 6, 2, 11, 15, 12, 9, 7, 3, 10, 5, 0,
	15, 12, 8, 2, 4, 9, 1, 7, 5, 11, 3, 14, 10, 0, 6, 13},

	{15, 1, 8, 14, 6, 11, 3, 4, 9, 7, 2, 13, 12, 0, 5, 10,
	3, 13, 4, 7, 15, 2, 8, 14, 12, 0, 1, 10, 6, 9, 11, 5,
	0, 14, 7, 11, 10, 4, 13, 1, 5, 8, 12, 6, 9, 3, 2, 15,
	13, 8, 10, 1, 3, 15, 4, 2, 11, 6, 7, 12, 0, 5, 14, 9},

	{10, 0, 9, 14, 6, 3, 15, 5, 1, 13, 12, 7, 11, 4, 2, 8,
	13, 7, 0, 9, 3, 4, 6, 10, 2, 8, 5, 14, 12, 11, 15, 1,
	13, 6, 4, 9, 8, 15, 3, 0, 11, 1, 2, 12, 5, 10, 14, 7,
	1, 10, 13, 0, 6, 9, 8, 7, 4, 15, 14, 3, 11, 5, 2, 12},

	{7, 13, 14, 3, 0, 6, 9, 10, 1, 2, 8, 5, 11, 12, 4, 15,
	13, 8, 11, 5, 6, 15, 0, 3, 4, 7, 2, 12, 1, 10, 14, 9,
	10, 6, 9, 0, 12, 11, 7, 13, 15, 1, 3, 14, 5, 2, 8, 4,
	3, 15, 0, 6, 10, 1, 13, 8, 9, 4, 5, 11, 12, 7, 2, 14},

	{2, 12, 4, 1, 7, 10, 11, 6, 8, 5, 3, 15, 13, 0, 14, 9,
	14, 11, 2, 12, 4, 7, 13, 1, 5, 0, 15, 10, 3, 9, 8, 6,
	4, 2, 1, 11, 10, 13, 7, 8, 15, 9, 12, 5, 6, 3, 0, 14,
	11, 8, 12, 7, 1, 14, 2, 13, 6, 15, 0, 9, 10, 4, 5, 3},

	{12, 1, 10, 15, 9, 2, 6, 8, 0, 13, 3, 4, 14, 7, 5, 11,
	10, 15, 4, 2, 7, 12, 9, 5, 6, 1, 13, 14, 0, 11, 3, 8,
	9, 14, 15, 5, 2, 8, 12, 3, 7, 0, 4, 10, 1, 13, 11, 6,
	4, 3, 2, 12, 9, 5, 15, 10, 11, 14, 1, 7, 6, 0, 8, 13},

	{4, 11, 2, 14, 15, 0, 8, 13, 3, 12, 9, 7, 5, 10, 6, 1,
	13, 0, 11, 7, 4, 9, 1, 10, 14, 3, 5, 12, 2, 15, 8, 6,
	1, 4, 11, 13, 12, 3, 7, 14, 10, 15, 6, 8, 0, 5, 9, 2,
	6, 11, 13, 8, 1, 4, 10, 7, 9, 5, 0, 15, 14, 2, 3, 12},

	{13, 2, 8, 4, 6, 15, 11, 1, 10, 9, 3, 14, 5, 0, 12, 7,
	1, 15, 13, 8, 10, 3, 7, 4, 12, 5, 6, 11, 0, 14, 9, 2,
	7, 11, 4, 1, 9, 12, 14, 2, 0, 6, 10, 13, 15, 3, 5, 8,
	2, 1, 14, 7, 4, 10, 8, 13, 15, 12, 9, 0, 3, 5, 6, 11}
    };

    sbpb6 fixed6bits;
    fbpb4R r;
    fbpb4 fixedr;

    fixed6bits = (((t6bits >> 0) & 01) << 5) +
	(((t6bits >> 1) & 01) << 3) +
	(((t6bits >> 2) & 01) << 2) +
	(((t6bits >> 3) & 01) << 1) +
	(((t6bits >> 4) & 01) << 0) +
	(((t6bits >> 5) & 01) << 4);

    r = S[tableno][fixed6bits];

    fixedr = (((r >> 3) & 01) << 0) +
	(((r >> 2) & 01) << 1) +
	(((r >> 1) & 01) << 2) +
	(((r >> 0) & 01) << 3);

    return (fixedr);
}

void
#ifndef CRYPT_64BIT
init (tableno, lowptr, highptr)
    sbpb24 *lowptr, *highptr;
#else
init (tableno, bothptr)
    sbpb24 *bothptr;
#endif
    unsl tableno;
{

    static unsb P[] =
    {
	15, 6, 19, 20,
	28, 11, 27, 16,
	0, 14, 22, 25,
	4, 17, 30, 9,
	1, 7, 23, 13,
	31, 26, 2, 8,
	18, 12, 29, 5,
	21, 10, 3, 24,
    };

    static unsb E[] =
    {
	31, 0, 1, 2, 3, 4,
	3, 4, 5, 6, 7, 8,
	7, 8, 9, 10, 11, 12,
	11, 12, 13, 14, 15, 16,
	15, 16, 17, 18, 19, 20,
	19, 20, 21, 22, 23, 24,
	23, 24, 25, 26, 27, 28,
	27, 28, 29, 30, 31, 0,
    };

    static obpb1 tmp32[32];
    static obpb1 tmpP32[32];
    static obpb1 tmpE[48];

    int i, j, k;
    int tablenoX4;
    reg sbpb24 low24, high24;

    tablenoX4 = tableno * 4;

    for (j = 0; j < 64; j++)
    {
	k = lookupS (tableno, j);

	for (i = 0; i < 32; i++)
	{
	    tmp32[i] = 0;
	}

	for (i = 0; i < 4; i++)
	{
	    tmp32[tablenoX4 + i] = (k >> i) & 01;
	}

	for (i = 0; i < 32; i++)
	{
	    tmpP32[i] = tmp32[P[i]];
	}

	for (i = 0; i < 48; i++)
	{
	    tmpE[i] = tmpP32[E[i]];
	}

	low24 = 0L;
	high24 = 0L;

	for (i = 0; i < 24; i++)
	{
	    low24 |= (unsigned long) tmpE[i] << i;
	}

	for (k = 0, i = 24; i < 48; i++, k++)
	{
	    high24 |= (unsigned long) tmpE[i] << k;
	}

        low24 = TF_TO_SIXBIT (low24);
        high24 = TF_TO_SIXBIT (high24);

#ifndef CRYPT_64BIT
#ifndef CRYPT_MIXED
	lowptr[j] = DO_SIZEFIX (low24);
	highptr[j] = DO_SIZEFIX (high24);
#else
	lowptr[j << 1] = DO_SIZEFIX (low24);
	highptr[j << 1] = DO_SIZEFIX (high24);
#endif
#else
        bothptr[j] = DO_SIZEFIX (low24) | (DO_SIZEFIX (high24) << 32);
#endif
    }
}

#ifdef CRYPT_128K

void
init_words (lowptr, highptr, wordptr)
    sbpb24 *lowptr, *highptr, *wordptr;
{
    reg int i, j;
    reg sbpb24 *ptr;

    ptr = wordptr;
#ifndef CRYPT_MIXED
    for (j = 0; j < 64; j++)
    for (i = 0; i < 64; i++)
        *ptr++ = lowptr[i] ^ highptr[j];
#else
    for (j = 0; j < 2 * 64; j += 2)
    for (i = 0; i < 2 * 64; i += 2) {
        *ptr = lowptr[i] ^ highptr[j]; ptr += 2;
    }
#endif
}

#endif

void
init_key ()
{
    reg int i, j, k;

    bzero(crypt_block, 64);
    for (i = 0; i < 8; i++)
    for (j = 127; j >= 0; j--) {
        for (k = 0; k < 7; k++) crypt_block[(i << 3) + k] = (j >> (6 - k)) & 1;
        fsetkey ( /* crypt_block */ );

	memcpy(BigKS[i][j], KS, KEYSIZE);
    }

    bzero(lastpw, sizeof(lastpw));
    memcpy(KS, BigKS, KEYSIZE);
}

int
init_des ()
{
    saltvalue = 0;

#ifndef CRYPT_64BIT
    init (0L, S0L, S0H);
    init (1L, S1L, S1H);
    init (2L, S2L, S2H);
    init (3L, S3L, S3H);
    init (4L, S4L, S4H);
    init (5L, S5L, S5H);
    init (6L, S6L, S6H);
    init (7L, S7L, S7H);

#ifdef CRYPT_128K
    init_words (S0L, S1L, SW0L);
    init_words (S0H, S1H, SW0H);
    init_words (S2L, S3L, SW1L);
    init_words (S2H, S3H, SW1H);
    init_words (S4L, S5L, SW2L);
    init_words (S4H, S5H, SW2H);
    init_words (S6L, S7L, SW3L);
    init_words (S6H, S7H, SW3H);
#endif

#else
    init (0L, S0);
    init (1L, S1);
    init (2L, S2);
    init (3L, S3);
    init (4L, S4);
    init (5L, S5);
    init (6L, S6);
    init (7L, S7);

#ifdef CRYPT_128K
    init_words (S0, S1, SW0);
    init_words (S2, S3, SW1);
    init_words (S4, S5, SW2);
    init_words (S6, S7, SW3);
#endif

#endif

    init_key ();

    return 0;
}
