; ROT13DEV.ASM - By La Fuerve Puba
; A very simple example of device drivers and how they work
; This particular driver doesn't do much, but it does (to me at least),
; show what a lengthy process writing a device driver is.
; It uses int 10h to output all the rot13 encoded characters, which it
; recieves from just about any standard input, thusly:
; COPY WEENER.TXT ROT13
; or,
; TYPE WEENER.TXT > ROT13
;
; There is no read function, so COPY ROT13 WEENER.TXT wouldn't work,
; however, if you wanted to make it perform that function, I'm sure
; that it would be possible.
;
; Make sure never to try to write to any files with the name rot13.*,
; or you'll get an error.  And furthermore, never try to read from
; the rot13 device, at all, trust me...
;
; A lot of work needs to be done, but it's not as if a rot13 device
; is going to revolutionize the world or anything.
;
; I'm sure you've seen similar code before, most likely the same stuff
; that I have.  Not like originality matters, there's only one way to
; write a device driver anyways.
;
; This code is public domain.  Do whatever the hell you want with it.
; If you want to get ahold of me for some reason, try me at
; fuerve@iswnet.com
;
; Assemble and link however you want, but you'll need exe2bin, as tlink
; won't generate a .COM with an org of 0h.  Then, just rename
; ROT13DEV.BIN to ROT13DEV.SYS
;
; Ok, on with the show

; EQUATES

; This set contains the values in the Request Packet given to the device
; driver for every call
; The actual structure is as follows:
;    rhLength       db ?         ; Length of record, in bytes
;    rhUnit         db ?         ; Unit number, for block devices only
;                                ; gives a 0-based drive number with
;                                ; which to work
;    rhFunction     db ?         ; Which function to call
;    rhStatus       dw ?         ; Status for return to calling program
;    rhReserved     db 8 dup(?)  ; 8 little nothings wasting space
;    There are occasionally extra fields, for specific functions
;    These will be shown below
;
; Diagram of rhStatus
;|15|14|13|12|11|10|09|08|07|06|05|04|03|02|01|00|
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |_ |-
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |____ |
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |_______ |
;  |  |  |  |  |  |  |  |  |  |  |  |  |__________ | Error codes
;  |  |  |  |  |  |  |  |  |  |  |  |_____________ | (see below)
;  |  |  |  |  |  |  |  |  |  |  |________________ |
;  |  |  |  |  |  |  |  |  |  |___________________ |
;  |  |  |  |  |  |  |  |  |______________________ |_
;  |  |  |  |  |  |  |  |_________________________ Operation completed
;  |  |  |  |  |  |  |____________________________ Device busy (set by
;  |  |  |  |  |  |                                Input status, Output
;  |  |  |  |  |  |                                status, and Removable
;  |  |  |  |  |  |                                media
;  |  |  |  |  |  |_______________________________ Reserved
;  |  |  |  |  |__________________________________ Reserved
;  |  |  |  |_____________________________________ Reserved
;  |  |  |________________________________________ Reserved
;  |  |___________________________________________ Reserved
;  |______________________________________________ Error bit
;                                                  If this bit is set,
;                                                  the error code in
;                                                  bits 0 - 7 are set to
;                                                  the applicable error
;
;
rhLength        EQU  <si>
rhUnit          EQU  <si + 1>
rhFunction      EQU  <si + 2>
rhStatus        EQU  <si + 3>
rsReserved      EQU  <si + 5>


B_Lower_Case    EQU  97    ; Lowest lower case ASCII character
U_Lower_Case    EQU  122   ; Highest lower case ASCII character
B_Upper_Case    EQU  65
U_Upper_Case    EQU  90

STDIN           EQU   0    ; stdin file handle

; Error codes

WP_Violation    EQU  00h   ; Write Protect Violation
Unknown_Unit    EQU  01h   ; Unknown unit
Not_Ready       EQU  02h   ; Drive not ready
Unknown_Cmd     EQU  03h   ; Unknown command
CRC_Error       EQU  04h   ; CRC Error
ILFDRS          EQU  05h   ; "Incorrect length for drive-request structure"
Seek_Error      EQU  06h   ; Seek error
Unknown_Media   EQU  07h   ; Unknown media
Sec_Not_Found   EQU  08h   ; Sector not found
Paper_Out       EQU  09h   ; Printer out of paper
Write_Fault     EQU  0Ah   ; This is microsoft's idea of an
                           ; all-encompassing and informative error
                           ; system designed for maximum safety and
                           ; efficiency
Read_Fault      EQU  0Bh   ; More of the same
Gen_Failure     EQU  0Ch   ; And this one epitomizes them all
Inv_Change      EQU  0Fh   ; Invalid disk change
                           ; 0Dh and 0Eh were "reserved"...
                           ; I hate that...

_TEXT           SEGMENT WORD PUBLIC 'CODE'
                ASSUME  CS:_TEXT, DS:_TEXT, ES:NOTHING
                ORG     0

; Device driver header
; The actual structure for a device driver header is as follows
;    dhLink                 dd ?       ; Link to the next driver
;    dhAttributes           dw ?       ; Attributes for device
;    dhStrategy             dw ?       ; Offset of strategy code
;    dhInterrupt            dw ?       ; Offset of interrupt code
;    dhNameorUnits          db '????????' ; Name of device (for
;                                         ; character devices), or
;                                         ; 1 byte value with the
;                                         ; number of block devices
;                                         ; supported.  If it's a
;                                         ; character device, the name
;                                         ; must be padded out to 8
;                                         ; bytes with spaces
;
; Diagram of dhAttributes
;    For character devices
;|15|14|13|12|11|10|09|08|07|06|05|04|03|02|01|00|
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |_ Standard input device
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |____ Standard output device
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |_______ NUL device
;  |  |  |  |  |  |  |  |  |  |  |  |  |__________ Clock device
;  |  |  |  |  |  |  |  |  |  |  |  |_____________ Fast char output
;  |  |  |  |  |  |  |  |  |  |  |________________ Reserved (blah)
;  |  |  |  |  |  |  |  |  |  |___________________ Generic IOCTL support
;  |  |  |  |  |  |  |  |  |______________________ Query IOCTL support
;  |  |  |  |  |  |  |  |_________________________ Reserved
;  |  |  |  |  |  |  |____________________________ Reserved
;  |  |  |  |  |  |_______________________________ Reserved
;  |  |  |  |  |__________________________________ Open and Close Device support
;  |  |  |  |_____________________________________ Reserved
;  |  |  |________________________________________ Output until busy support
;  |  |___________________________________________ IOCTL Read/Write support
;  |______________________________________________ 1 = Character device
;
; Notes: You can replace the current STDIN, STDOUT, and CLOCK devices
; by setting the respective bits.  You cannot, however, replace NUL, for
; some reason.  Go figure.
; If bit 4 is set, DOS will call int 29h with a character value in AL
; whenever the device is written to.  So, in order to use fast character
; output, you have to install an interrupt handler for int 29h.
; Bit 11 toggles whether your driver supports Open Device, Close Device,
; and Removable Media.  Character devices can support open and close, but
; only block devices can support removable media.
; Bit 15, if set, means that the device is a character device, and if unset,
; means that it's a block device.
;
;    For block devices
;|15|14|13|12|11|10|09|08|07|06|05|04|03|02|01|00|
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |_ Reserved
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |____ Driver supports 32-bit
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |        sector addressing
;  |  |  |  |  |  |  |  |  |  |  |  |  |  |_______ Reserved
;  |  |  |  |  |  |  |  |  |  |  |  |  |__________ Reserved
;  |  |  |  |  |  |  |  |  |  |  |  |_____________ Reserved
;  |  |  |  |  |  |  |  |  |  |  |________________ Reserved
;  |  |  |  |  |  |  |  |  |  |___________________ Driver supports Generic
;  |  |  |  |  |  |  |  |  |                       IOCTL, Get Logical Device,
;  |  |  |  |  |  |  |  |  |                       and Set Logical Device
;  |  |  |  |  |  |  |  |  |______________________ Support for Query IOCTL
;  |  |  |  |  |  |  |  |_________________________ Reserved
;  |  |  |  |  |  |  |____________________________ Reserved
;  |  |  |  |  |  |_______________________________ Reserved
;  |  |  |  |  |__________________________________ Support for Open Device,
;  |  |  |  |                                      Close Device, and
;  |  |  |  |                                      Removable Media
;  |  |  |  |_____________________________________ Reserved
;  |  |  |________________________________________ Driver requires DOS to
;  |  |                                            give it the FAT for the
;  |  |                                            drive when issuing the
;  |  |                                            Build BPB command
;  |  |___________________________________________ Support for IOCTL Read/Write
;  |______________________________________________ 0 = Block Device
;
; Notes:
; If your driver supports 32-bit sector addressing, set bit 1
; This will set the rwHugeStartSec in the Read and Write functions
; If you want DOS to load a copy of the FAT on a call to Build BPB,
; Then set bit 13.  This is for when you have no way of determining the
; media by any other means.

; Device header for ROT13
                dd      -1      ; Link to next driver, set by dos
                                ; always set to FFFFFFFF by default, let
                                ; dos take care of it later
                dw      0A800h
                                ; Device attributes
                dw      offset Strategy
                                ; Strategy, called prior to calling the
                                ; interrupt
                dw      offset Interrupt
                                ; The interrupt does the gruntwork
                                ; the term interrupt in this case
                                ; is used rather liberally
                db      'ROT13   '
                                ; Since it's a character device, it
                                ; needs a name

; Other data

Request_Address dw      0000,0000
                                ; The segment and offset of the request
                                ; packet are stored here by the strategy
                                ; code before the interrupt is called

Num_Chars       dw      0       ; Number of characters to process
Buffer_Complete dw      0       ; How much of the buffer has been
                                ; processed

Vid_Page        db     -1       ; Current video page


; Table of device commands
Command_Table   dw      offset Init     ; Function 00h, Initialize driver
                dw      offset MediaChk ; Function 01h, Media Check
                dw      offset Build_BPB; Function 02h, Build BPB
                dw      offset IOCTL_RD ; Function 03h, IOCTL Read
                dw      offset Read     ; Function 04h, Read
                dw      offset ND_Read  ; Function 05h, Nondestructive
                                        ; read
                dw      offset In_Stat  ; Function 06h, Input status
                dw      offset In_Flush ; Function 07h, Input flush
                dw      offset Write    ; Function 08h, Write
                dw      offset WriteVRFY; Function 09h, Write with verify
                dw      offset Out_Stat ; Function 0Ah, Output status
                dw      offset Out_Flush; Function 0Bh, Output flush
                dw      offset IOCTL_WR ; Function 0Ch, IOCTL Write
                dw      offset Open     ; Function 0Dh, Open Device
                dw      offset Close    ; Function 0Eh, Close Device
                dw      offset Rmvable  ; Function 0Fh, Removable media
                dw      offset Out_Busy ; Function 10h, Output until busy
                dw      offset Invalid  ; Function 11h, Nonexistant
                dw      offset Invalid  ; Function 12h, Nonexistant
                dw      offset IOCTL    ; Function 13h, Generic IOCTL
                dw      offset Invalid  ; Function 14h, Nonexistant
                dw      offset Invalid  ; Function 15h, Nonexistant
                dw      offset Invalid  ; Function 16h, Nonexistant
                dw      offset Get_Dev  ; Function 17h, Get logical device
                dw      offset Set_Dev  ; Function 18h, Set logical device
                dw      offset IOCTL_Q  ; Function 19h, IOCTL Query

; Strategy routine
Strategy        PROC    FAR
                mov     CS:Request_Address[0], bx
                mov     CS:Request_Address[2], es
                                        ; Set segment and offset of
                                        ; request packet
                                        ; DOS calls the strategy routine
                                        ; with the seg:off pair in
                                        ; ES:BX
                ret
Strategy        ENDP

; Device interrupt routine
Interrupt       PROC    FAR
                ; Push everything
                pushf
                push    ax
                push    bx
                push    cx
                push    dx
                push    si
                push    di
                push    ds
                push    es
                push    bp

                ; Set our data segment to the code segment,
                ; since it's all stored in one place
                push    cs
                pop     ds

                ; Set ES:SI to equal the seg:off of the request packet
                mov     si, CS:Request_Address[0]
                mov     ax, CS:Request_Address[2]
                mov     es, ax

                ; Get the function we're performing
                mov     bl, es:[rhFunction]
                ; Test to see if the command is out of range
                cmp     bl, 19h
                jbe     Command_OK
                ; If not, return an error
                call    Invalid
                ; And return to the calling program
                jmp     Interrupt_End

                ; If the command is within range
Command_OK:
                ; Command = offset of Command_Table + BX

                ; BX = BX * 2, to give the exact byte position
                ; of the offset word

                mov     bh, 0
                shl     bx, 1
                ; Call the command
                call    word ptr [bx + Command_Table]

; Finished executing the command
; Now wrap it up and return to the calling program

Interrupt_End:
                ; If AL = 0, there's no error
                cmp     al, 0
                je      No_Error

                ; If there is an error, turn on the error bit
                or      ah, 10000000b

No_Error:
                ; Regardless of whether there's an error, turn on
                ; the Operation Complete bit
                or      ah, 00000001b

                ; Now stuff the return status back into the request
                ; packet for the sake of the calling program
                mov     es:[rhStatus], ax

                ; Tie up some loose ends
                pop     bp
                pop     es
                pop     ds
                pop     di
                pop     si
                pop     dx
                pop     cx
                pop     bx
                pop     ax
                popf

                ; And exit out
                ret

Interrupt       ENDP
; Thus ends the interrupt procedure

; Begin device function code
; Since this driver is fairly simplistic, most of the functions will
; just return as soon as they're called.

; Function 01h, Media Check
; Checks to see if the disk has been changed
; In addition to the default Request Packet values, there are some
; additional pieces of information that might be necessary, and
; that are given along with the packet, directly after the 8 reserved
; bytes.
;
; rhMedia_ID    db ?    ; Media Descriptor
;    Some common ones:
;       0F0h     -   1.44MB 3.5", 2.88MB 3.5", 1.2MB 5.25"
;                    various others
;       0F8h     -   Hard Disk of any type
;       0F9h     -   720KB 3.5", 1.2MB 5.25"
;       0FAh     -   320KB 5.25"
;       0FBh     -   640KB 3.5"
;       0FCh     -   180KB 5.25"
;       0FDh     -   360KB 5.25", and 8" disks
;       0FFh     -   320KB 5.25"
;
; rhReturn      db ?    ; Return value
;    On return from your function, put one of these values
;       0FFh     -   Disk has been changed
;        00h     -   Can't tell whether it's been changed
;        01h     -   It hasn't been changed
;
; rhVolume_ID   dd ?    ; Pointer to the volume ID for the disk
;    It's up to you to set, read, and store volume IDs for the disk
;    you're coding a driver for.
;    And so, upon return from your function, you have to set a seg:off
;    dword pointing to the address of an ASCIIZ volume label.  If there
;    is no other label, you have to return a pointer to the string
;    "NO NAME"
;
;    DOS performs this function to see whether it wants to go on with
;    its disk functions.  If it finds that the disk has changed, then
;    it'll shitcan all the associated buffers and let the user wonder
;    why.  FUN!

MediaChk        PROC    NEAR

                ; Since this isn't a block driver, we just return from
                ; this function

                mov     ax, 0      ; Set the error code

                ; And get the hell out
                ret
MediaChk        ENDP

; Function 02h, Build BPB (BIOS Parameter Block)
; Builds a BPB for the disk and returns it
; This is called every time function 01h reports a disk change,
; or whenever else dos feels like it
;
; The extra Request Packet values are as follows:
;    rhMedia_ID     db ? ; Same as for Media Check
;    rhFAT_Sector   dd ? ; seg:off address of a 512 byte buffer in
;                        ; which, if bit 13 in the Device Attributes
;                        ; was set, is a copy of the first sector of the
;                        ; FAT of the disk that the driver is to work on
;    rhBPB_Address  dd ? ; The Build BPB function must set this to point
;                        ; to a valid BPB (You can find a description of
;                        ; the BPB in HELPPC, or in various books, such
;                        ; as the one I have sitting on my lap as I
;                        ; write this :) )

Build_BPB       PROC    NEAR

                ; Again, we do nothing but get the hell out
                mov     ax, 0
                ret
Build_BPB       ENDP

; Function 03h, IOCTL Read
; This function allows IOCTL to read data from your driver, and works
; with both character and block devices
;
; The extra parameters in the Request Packet are as follows:
;    rhData       db ?      ; Not used (says the giant wasteful beast)
;    rhBuffer     dd ?      ; Address of buffer to write the data to
;    rhNum_Bytes  dw ?      ; On entry, it equals the number of bytes
;                           ; requested by IOCTL.  On exit, set it to
;                           ; the actual number of bytes read from the
;                           ; device.

IOCTL_RD        PROC    NEAR
                ; Another one which I will gladly pass up the chance
                ; to support
                mov     ax, 0
                ret
IOCTL_RD        ENDP

; Function 04h, Read
; This function is a required function for both character and block
; drivers.  That doesn't mean you have to have it do anything, of
; course.  It just means that it has to be there, in some form.
;
; There are 17 bytes of extra crap stacked on this one:
;    rhMedia_ID      db ?     ; Same as the rest
;    rhBuffer        dd ?     ; Pointer to buffer to put stuff in
;    rhBytes_Secs    dw ?     ; Number of bytes/sectors to read
;                             ; On exit, set to the actual number read
;    rhStart_Sec     dw ?     ; Number of starting sector
;    rhVolume_ID     dd ?     ; On exit, should point to an ASCIIZ
;                             ; volume label
;    rhHugeStartSec  dd ?     ; 32-bit starting sector, for 32 bit I/O
;                             ; only valid if rhStart_Sec = 0FFFFh

Read            PROC    NEAR
                ; If you really wanted to be cool, implement                
                ; being able to do it straight from stdin,
                ; like COPY CON

                mov     ax, 0
                ret

Read            ENDP

; Function 05h, Nondestructive Read
; This function reads the next character from the given character device
; without removing the character from the buffer
;
; Extra stuff:
;    rhChar     db ?    ; On exit, set to the value of the character
;                       ; read in from the device

ND_Read         PROC    NEAR
                ; Set Device Busy bit to indicate no characters waiting
                ; If there were characters, we'd leave it be
                or      ah, 00000010b
                ; And leave
                ret
ND_Read         ENDP

; Function 06h, Input Status
; This function lets dos know whether there are any characters buffered
; If there are no characters, set the busy bit, otherwise clear it
; No extra parameters

In_Stat         PROC    NEAR
                ; Usually, since there are no character's, we'd
                ; have to set the busy bit, but since we never plan
                ; on returning characters, and since dos'll sit and
                ; wait for us to do so, we just return
                mov     ax, 0
                ret
In_Stat         ENDP

; Function 07h, Input Flush
; Flushes all input buffers, and shitcans any current read operation
; No extra parameters

In_Flush        PROC    NEAR
                ; Just return
                mov     ax, 0
                ret
In_Flush        ENDP

; Function 08h, Write
; Write the buffer to the specific device
; Extra parameters are the same as for read

Write           PROC    NEAR
                ; Finally, we actually do something

                ; First, save some stuff
                push    es
                push    si

                ; Find out what video page we're writing to
                call    Get_Page

                ; Get segment and offset of buffer to write
                ; As well as the length of the buffer
                mov     ax, word ptr es:[si + 0Eh]  ; Offset of buffer
                mov     bx, word ptr es:[si + 10h]  ; Segment of buffer
                mov     cx, word ptr es:[si + 12h]  ; Length of buffer

                ; Store the values we need, and get ready to do the work
                mov     cs:[Num_Chars], cx
                mov     cs:[Buffer_Complete], 0

                ; In case there are no characters in the buffer
                jcxz    Write_Finished

                ; Set up the rest of our values
                mov     es, bx
                mov     si, ax
                ; ES:SI points to the buffer

                ; Call the rot13 function, which will encode and print
                ; the entire buffer
                call    Rot13

Write_Finished:
                ; Get ES:SI back
                pop     si
                pop     es

                ; Set the error flags to 0
                mov     ax, 0

                ; Check to see if we completed the task
                mov     cx, cs:[Buffer_Complete]
                mov     word ptr es:[si + 12h], cx

                ; Does the number of characters written equal the
                ; number of characters in the buffer?
                cmp     cx, cs:[Num_Chars]

                ; If so, quit
                je      Write_End

                ; Otherwise...
                mov     al, Write_Fault

Write_End:
                ; Finished with everything
                ret
Write           ENDP


; Function 09h, Write with Verify
; Write to device, and read back to verify
; Extra parameters the same as Write

WriteVRFY       PROC    NEAR
                ; Nothing to do here
                mov     ax, 0
                ret
WriteVRFY       ENDP


; Function 0Ah, Output Status
; Reports whether there are any characters in the output buffer
; For character devices only
Out_Stat        PROC    NEAR
                ; A nice simple device driver doesn't need such things
                mov     ax, 0
                ret
Out_Stat        ENDP

; Function 0Bh, Output Flush
; Nukes every write operation and flushes all associated output buffers
; Character devices only

Out_Flush       PROC    NEAR
                ; Don't need this
                mov     ax, 0
                ret
Out_Flush       ENDP

; Function 0Ch, IOCTL Write
; Allows IOCTL to write the device
; Extra parameters are the same as for IOCTL Read

IOCTL_WR        PROC    NEAR
                ; No IOCTL support
                mov     ax, 0
                ret
IOCTL_WR        ENDP

; Function 0Dh, Open Device
; Gives the driver a chance to set itself up
Open            PROC    NEAR
                ; Just set the video page
                call    Get_Page
                ; And we're done
                mov     ax, 0
                ret
Open            ENDP

; Function 0Eh, Close Device
; Gives the driver a chance to tie up loose ends when the device closes
Close           PROC    NEAR
                ; But our device is open 24 hours
                mov     ax, 0
                ret
Close           ENDP

; Function 0Fh, Removable Media
; Returns a value stating whether the drive is removable
; If so, clear the busy bit and set the low byte of rhStatus to
; one of the previously shown values, otherwise, set the busy bit
Rmvable         PROC    NEAR
                ; Remove THIS, buddy
                mov     ax, 0
                ret
Rmvable         ENDP

; Function 10h, Output Until Busy
; Feeds the device until it signals that it can't take any more
; Extra parameters are the same as for IOCTL Read
; Character devices only

Out_Busy        PROC    NEAR
                ; Same as a regular write
                ; Just set the busy flag to get it to shut up

                ; Just call the write function
                call    Write

                ; And turn off any errors
                or      ah, 00000010b
                and     al, 00000000b

                ; And quit
                ret
Out_Busy        ENDP

; Function 13h, Generic IOCTL
; Instructs the driver from inside the Request Packet
;
; Extra Parameters:
;    rhCategory      db ?       ; Device category
;    rhMinor_Code    db ?       ; Minor code from IOCTL
;    rhReserved_2    dd ?       ; Don't ask me
;    rhIOCTL_Data    dd ?       ; Pointer to the data
;
; Notes on rhCategory:
;    01h   -   Serial device
;    03h   -   Console
;    05h   -   Printer
;    08h   -   Disk
;    You can also allow for custom values, so long as you provide
;    support
;
; Notes on rhMinor_Code:
;    For serial, console, and printer drivers:
;       45h   -   Set Iteration Count
;       4Ah   -   Select Code Page
;       4Ch   -   Start Code Page Prepare
;       4Dh   -   End Code Page Prepare
;       65h   -   Get Iteration Count
;       6Ah   -   Query Selected Code Page
;       6Bh   -   Query Code Page Prepare List
;    For disk drivers:
;       40h   -   Set Device Parameters
;       41h   -   Write Track on Logical Drive
;       42h   -   Format Track on Logical Drive
;       46h   -   Set Media ID
;       60h   -   Get Device Parameters
;       61h   -   Read Track on Logical Drive
;       62h   -   Verify Track on Logical Drive
;       66h   -   Get Media ID
;       68h   -   Sense Media Type
;
IOCTL           PROC    NEAR
                ; Luckily for me, I could give a shit
                mov     ax, 0
                ret
IOCTL           ENDP

; Function 17h, Get Logical Device
; Gets the current block device the driver is servicing
; On exit: Set rhUnit to the 1-based drive number
; (1 = A:, 2 = B:, et cetera)
Get_Dev         PROC    NEAR
                ; But who needs that
                mov     ax, 0
                ret
Get_Dev         ENDP

; Function 18h, Set Logical Dev0ice
; Sets the block device for the driver to service
; rhUnit is a 0-based drive number
Set_Dev         PROC    NEAR
                ; No block devices here
                mov     ax, 0
                ret
Set_Dev         ENDP

; Function 19h, IOCTL Query
; Determines whether a specific minorcode is supported by the driver
; The extra parameters are the same as Generic IOCTL
IOCTL_Q         PROC    NEAR
                ; And thank the gods that it has nothing to do with
                ; this, whatsoever
                mov     ax, 0
                ret
IOCTL_Q         ENDP

; All the other functions
; Gives an error and tells the calling program to fuck off
Invalid         PROC    NEAR
                ; Set the error code
                mov     ax, Unknown_Cmd
                ; And return
                ret
Invalid         ENDP

; Get video page
Get_Page        PROC    NEAR
                ; Get the video page
                mov     ah, 0Fh
                int     10h
                ; Display page in BH
                mov     cs:[Vid_Page], bh
                ; And return to the calling function
                ret
Get_Page        ENDP



; Rot13 encode and print the text
Rot13           PROC    NEAR

                ; Get the video page
                mov     bh, cs:[Vid_Page]
Rot13_Loop:
                ; NOTE: this whole routine is probably pretty flaky

                ; Save the loop counter
                push    cx

                ; Get the next character from the buffer
                mov     al, byte ptr es:[si]
                ; And increment SI for the next character
                inc     si

                ; Begin rot13 modifications

Rot13_Upper_Case:
                ; Is it less than 'A'
                cmp     al, B_Upper_Case
                ; If so, fuck it
                jl      Rot13_Done

                ; Is it greater than 'Z'
                cmp   al, U_Upper_Case
                ; If so, maybe it's lower case
                jg    Rot13_Lower_Case

                ; Otherwise, encode it
                cmp   al, (B_Upper_Case + 13)
                jge   Rot13_Decrement
                jmp   Rot13_Increment

Rot13_Lower_Case:
                ; Is it less than 'a'
                cmp   al, B_Lower_Case
                ; If so, quit
                jl    Rot13_Done
                ; And if it's higher than 'z'...
                cmp   al, U_Lower_Case
                ; Same deal
                jg    Rot13_Done

                ; Otherwise, encode it
                cmp   al, (B_Lower_Case + 13)
                jge   Rot13_Decrement
                ; If it's not greater than or equal to 'n', go straight
                ; to Rot13_Increment

Rot13_Increment:
                ; Increment the character
                add   al, 13
                jmp   Rot13_Done

Rot13_Decrement:
                ; Decrement the character
                sub   al, 13
                ; Go straight to Rot13_Done

Rot13_Done:
                ; BIOS write character in teletype mode
                mov   ah, 0Eh
                int   10h

                ; Increase the number of characters completed
                mov   cx, cs:[Buffer_Complete]
                inc   cx
                mov   cs:[Buffer_Complete], cx

                ; Grab our loop counter
                pop   cx

                ; And loop until the entire buffer has been processed
                loop  Rot13_Loop

                ; And now we're done
                ret
Rot13           ENDP

; Function 00h, Initialize the bloody device driver
; Does exactly what you think it does, but probably slower and less
; efficiently
; Extra parameters:
;    rhUnits       db ?         ; Number of units supported by driver
;                               ; in case you want to support more than
;                               ; one drive
;    rhEndAddress  dd ?         ; Seg:off of the end of the driver
;                               ; itself.  Usually set to the beginning
;                               ; of the Init function.  Set this to get
;                               ; rid of all the shit you won't need
;                               ; need after it's all loaded
;    rhParmAddress dd ?         ; On entry: points to the DEVICE line
;                               ; in CONFIG.SYS
;                               ; On exit: points to the array of
;                               ; BPB structures for all the drives
;                               ; your driver supports
;    rhDrive_Num   db ?         ; 0-based number of first supported
;                               ; drive
;    rhMessageFlag dw ?         ; Boolean flag: whether an error message
;                               ; will be displayed or not
;                               ; You must also set rhStatus to indicate
;                               ; load failure
Init            PROC    NEAR

               ; Prepare to do some neato device driver stuff
                push    si

                ; Print our neat little string
                mov     si, offset Load_Message
                call    Print_String

                pop     si

                ; Set rhEndAddress
                mov     word ptr es:[si + 0Eh], offset Init  ; offset
                mov     word ptr es:[si + 10h], cs           ; segment

                mov     ax, 0

                ret
Init            ENDP

; Print an ASCIIZ string
; On entry: String pointer in DS:SI
Print_String    PROC    NEAR
                ; First, get the video page
                mov     bh, [Vid_Page]

Print_Loop:
                ; Get the next character
                mov     al, [si]
                inc     si

                ; End of the string?
                cmp     al, 0

                ; If so, exit
                je      Print_Done

                ; Otherwise, call int 10h function 0Eh
                ; Print character
                mov     ah, 0Eh
                int     10h

                ; And loop, until string is complete
                jmp     Print_Loop

Print_Done:     ; Finished printing the string
                ; So quit
                ret

Print_String    ENDP

; Message to display when loading (ASCIIZ)
Load_Message    db      0Dh, 0Ah
                db      '----------------------------------------------',0Dh,0Ah
                db      'ROT13 Device Driver Loaded and Enabled',0Dh,0Ah
                db      'This driver is public domain',0Dh,0Ah
                db      '   by: La Fuerve Puba',0Dh,0Ah
                db      '----------------------------------------------',0Dh,0Ah
                db      0

_TEXT           ENDS
                END
