////////////////////////////////////////////////////////////
//
//  BMP_QUAN.H - MS-Windows BMP Color Quantization Class
//               Include File
//
//  Version:    1.00B
//
//  History:    94/08/23 - Version 1.00A release.
//              94/11/11 - Version 1.00B release.
//
//  Compilers:  Microsoft Visual C/C++ Professional V1.5
//              Borland C++ Version 4.0
//
//  Author:     Ian Ashdown, P.Eng.
//              byHeart Software Limited
//              620 Ballantree Road
//              West Vancouver, B.C.
//              Canada V7S 1W3
//              Tel. (604) 922-6148
//              Fax. (604) 987-7621
//
//  Copyright 1994 byHeart Software Limited
//
////////////////////////////////////////////////////////////

#ifndef _BMP_QUAN_H
#define _BMP_QUAN_H

#include "dib_dos.h"
#include "oct_quan.h"

// Maximum number of quantized colors
//
// NOTE: MS-Windows reserves 20 static colors for its
//       default palette. Limiting the BMP file color
//       palette to 236 colors ensures that these colors
//       will not be affected when the BMP file is
//       displayed.

static const int BMP_MaxColor = 236;

class BMP_Quantize : public OctQuant
{
  private:
    char *pin_name;     // Input file name
    char *pout_name;    // Output file name

    int num_color;      // Number of palette colors

    size_t in_len;      // Input scan line length
    size_t out_len;     // Output scan line length

    BYTE *pin_scan;     // Input scan line buffer pointer
    BYTE *pout_scan;    // Output scan line buffer pointer

    DIB_FileHeader f_header;    // DIB file header
    DIB_InfoHeader i_header;    // DIB info header
    DIB_Palette dib_pal[256];   // DIB color palette

    DWORD in_offset;    // Input file bitmap offset

    FILE *pin_file;     // Input file pointer
    FILE *pout_file;    // Output file pointer

    BOOL GetPixel( int col, int row, OctColor &c )
    {
      BYTE *ppixel;     // Scan line buffer pixel pointer

      if (col == 0)
      {
        // Read in next scan line
        if (fread(pin_scan, sizeof(BYTE), in_len, pin_file)
            != in_len)
          return FALSE;
      }

      // Get pixel colors
      ppixel = pin_scan + col * 3;
      c.SetBlue(*ppixel++);
      c.SetGreen(*ppixel++);
      c.SetRed(*ppixel);

      return TRUE;
    }

    BOOL MapColors();

    BOOL SetPixel( int col, int row, BYTE color )
    {
      // Set pixel color
      *(pout_scan + col) = color;

      if (col == (i_header.biWidth - 1))
      {
        // Write scan line
        if (fwrite(pout_scan, sizeof(BYTE), out_len,
            pout_file) != out_len)
          return FALSE;
      }

      return TRUE;
    }

    // Reset input file pointer to beginning of bitmap
    BOOL Reset()
    {
      return (fseek(pin_file, in_offset, SEEK_SET) == 0) ?
          TRUE : FALSE;
    }

    // Scan line width in bytes must be multiple of four to
    // avoid memory segmentation problems
    size_t GetScanLineWidth( size_t num_pixel )
    { return (((num_pixel + 3) / 4) * 4); }

    void InitPalette();

    void ReportError( char *msg, char *var )
    {
      char temp_buff[128];      // Temporary buffer

      sprintf(temp_buff, msg, var);
      fprintf(stderr, "ERROR: %s\n", temp_buff);
    }

  public:
    BMP_Quantize() : OctQuant()
    {
      int i;    // Loop index

      pin_file = pout_file = NULL;
      pin_name = pout_name = NULL;
      pin_scan = pout_scan = NULL;
      num_color = 0;
      max_color = BMP_MaxColor;

      for (i = 0; i < 256; i++)
      {
        dib_pal[i].rgbBlue = (BYTE) 0;
        dib_pal[i].rgbGreen = (BYTE) 0;
        dib_pal[i].rgbRed = (BYTE) 0;
        dib_pal[i].rgbReserved = (BYTE) 0;
      }
    }

    BOOL Quantize( char *, char * );
};

#endif

