////////////////////////////////////////////////////////////
//
//  OCT_QUAN.H - Octree Color Quantization Class Include
//               File
//
//  Version:    1.00B
//
//  History:    94/08/23 - Version 1.00A release.
//              94/11/11 - Version 1.00B release.
//
//  Compilers:  Microsoft Visual C/C++ Professional V1.5
//              Borland C++ Version 4.0
//
//  Author:     Ian Ashdown, P.Eng.
//              byHeart Software Limited
//              620 Ballantree Road
//              West Vancouver, B.C.
//              Canada V7S 1W3
//              Tel. (604) 922-6148
//              Fax. (604) 987-7621
//
//  Copyright 1994 byHeart Software Limited
//
////////////////////////////////////////////////////////////

#ifndef _OCT_QUAN_H
#define _OCT_QUAN_H

#include "gen_defs.h"

// Maximum octree reduction level
static const int O_MaxDepth = 7;

// Number of quantized colors
static const int O_NumColor = 256;

class OctColor          // 24-bit RGB color model
{
  private:
    BYTE red;
    BYTE green;
    BYTE blue;

  public:
    BYTE GetBlue() { return blue; }
    BYTE GetGreen() { return green; }
    BYTE GetRed() { return red; }
    void SetBlue( BYTE b ) { blue = b; }
    void SetGreen( BYTE g ) { green = g; }
    void SetRed( BYTE r ) { red = r; }
};

class OctNode           // Octree node
{                       
  private:
    int level;          // Node level
    BOOL leaf_flag;     // Leaf flag
    BOOL mark_flag;     // Marked flag
    DWORD count;        // Pixel count
    struct
    {
      DWORD red;
      DWORD green;
      DWORD blue;
    }
    sum;                // Summed color value
    int index;          // Color palette index
    int children;       // Number of child nodes
    OctNode *pchild[8]; // Children node pointers
    OctNode *pnext;     // Next reducible node pointer
    OctNode *pprev;     // Previous reducible node pointer

    int TestBit(BYTE val, int index)
    { return ((val & (1 << index)) > index) ? 1 : 0; }

  public:
    OctNode( int node_level, BOOL leaf )
    {
      int i;    // Loop index

      level = node_level;
      leaf_flag = leaf;
      mark_flag = FALSE;
      count = 0L;
      index = 0;
      children = 0;
      sum.red = sum.green = sum.blue = 0L;

      for (i = 0; i < 8; i++)
        pchild[i] = NULL;

      pnext = pprev = NULL;
    };

    BOOL IsLeaf() { return leaf_flag; }
    BOOL IsMark() { return mark_flag; }
    
    DWORD GetCount() { return count; }

    OctColor GetColor()
    {
      OctColor temp;    // Temporary color

      temp.SetRed((BYTE) (sum.red / count));
      temp.SetGreen((BYTE) (sum.green / count));
      temp.SetBlue((BYTE) (sum.blue / count));

      return temp;
    }

    int GetIndex() { return index; }
    int GetLevel() { return level; }

    // Determine child node according to color
    int FindChild( OctColor &c )
    {
      int index;    // Child node pointer index

      // Determine child node pointer index
      index = TestBit(c.GetRed(), O_MaxDepth - level) << 2 |
          TestBit(c.GetGreen(), O_MaxDepth - level) << 1 |
          TestBit(c.GetBlue(), O_MaxDepth - level);

      return index;
    }

    OctNode *GetChild( int i ) { return pchild[i]; }
    OctNode *GetNext() { return pnext; }
    OctNode *GetPrev() { return pprev; }
    int GetNumChild() { return children; }

    // Add RGB color to node
    void AddColor( OctColor &c )
    {
      sum.red += (DWORD) c.GetRed();
      sum.green += (DWORD) c.GetGreen();
      sum.blue += (DWORD) c.GetBlue();

      count++;
    }

    void DecNumChild() { children--; }
    void IncNumChild() { children++; }
    void SetChild( int i, OctNode *pc ) { pchild[i] = pc; }
    void SetIndex( int i ) { index = i; }
    void SetLeaf( BOOL flag ) { leaf_flag = flag; }
    void SetMark( BOOL flag ) { mark_flag = flag; }
    void SetNext( OctNode *pn ) { pnext = pn; }
    void SetPrev( OctNode *pn ) { pprev = pn; }
};

class OctQuant          // Octree color quantization
{
  private:
    int leaf_level;     // Leaf level
    int num_leaf;       // Number of leaf nodes

    // Reducible node list pointers
    OctNode *prnl[O_MaxDepth +1];

  protected:
    int height;         // Bitmap height
    int width;          // Bitmap width
    int max_color;      // Maximum number of colors

    OctNode *proot;     // Octree root node pointer

    static OctColor Palette[O_NumColor];    // Color palette

    BOOL InsertNode( OctNode *, OctColor & );

    OctNode *MakeNode( int level )
    {
      BOOL leaf;    // Leaf node flag

      leaf = (level >= leaf_level) ? TRUE : FALSE;

      if (leaf == TRUE)
        num_leaf++;

      return new OctNode(level, leaf );
    }

    OctNode *GetReducible();

    // Quantize color
    int QuantizeColor( OctNode *pn, OctColor &c )
    {
      int c_index;      // Child node pointer index

      if ((pn->IsLeaf() == TRUE) || pn->GetLevel() ==
          leaf_level)
        return pn->GetIndex();
      else
      {
        c_index = pn->FindChild(c);

        return QuantizeColor(pn->GetChild(c_index), c);
      }
    }

    virtual BOOL GetPixel( int, int, OctColor & ) = 0;
    virtual BOOL SetPixel( int, int, BYTE ) = 0;

    void DeleteNode( OctNode * );
    void DeleteTree() { DeleteNode(proot); }
    void FillPalette( OctNode *, int * );

    // Add node to reducible list
    void MakeReducible( OctNode *pn )
    {
      int level;        // Node level
      OctNode *phead;   // List head node
      
      level = pn->GetLevel();
      phead = prnl[level];
      pn->SetNext(phead);
      if (phead != NULL)
        phead->SetPrev(pn);
      prnl[level] = pn;

      pn->SetMark(TRUE);
    }
    
    void ReduceTree();

  public:
    OctQuant()
    {
      int i;    // Loop index

      width = height = 0;
      num_leaf = 0;
      max_color = O_NumColor;
      leaf_level = O_MaxDepth + 1;

      // Clear the reducible node list pointers
      for (i = 0; i < leaf_level; i++)
        prnl[i] = NULL;

      // Clear the color palette
      for (i = 0; i < O_NumColor; i++)
      {
        Palette[i].SetRed((BYTE) 0);
        Palette[i].SetGreen((BYTE) 0);
        Palette[i].SetBlue((BYTE) 0);
      }

      proot = NULL;
    }

    BOOL BuildTree();
    
    virtual BOOL MapColors() = 0;
    virtual void InitPalette() = 0;
};

#endif

