////////////////////////////////////////////////////////////
//
//  WINGTEST.CPP - Winged Edge Boundary Representation Class
//                 Test Program
//
//  Version:    1.00B
//
//  History:    96/05/05 - Version 1.00A release.
//				97/04/24 - Version 1.00B release.
//
//  Compilers:  Microsoft Visual C/C++ Professional V5.0
//              Borland C++ Version 5.0
//
//  Author:     Ian Ashdown, P.Eng.
//              byHeart Consultants Limited
//              620 Ballantree Road
//              West Vancouver, B.C.
//              Canada V7S 1W3
//              Tel. (604) 922-6148
//              Fax. (604) 987-7621
//
//  Copyright 1996-1997 byHeart Consultants Limited
//
//  The following source code may be freely copied,
//  redistributed, and/or modified for personal use ONLY,
//  as long as the copyright notice is included with all
//  source code files.
//
////////////////////////////////////////////////////////////

#include "winged.h"

static Point3 TestPoint[5];
static WVert *VertPtr[5];
static WEdge *EdgePtr[6];
static WFace *FacePtr[3];

void DisplayEdgeData( WEdge * );
void DisplayFaceData( WFace * );
void DisplaySurfData( WSurf * );
void DisplayVertData( WVert * );

int main( int argc, char **argv )
{
  WSurf *psurf;         // Surface pointer

  // Allocate new surface
  psurf = new WSurf;

  // Initialize points array
  TestPoint[0].SetX(0.0);
  TestPoint[0].SetY(0.0);
  TestPoint[0].SetZ(0.0);
  TestPoint[1].SetX(1.0);
  TestPoint[1].SetY(0.0);
  TestPoint[1].SetZ(0.0);
  TestPoint[2].SetX(1.0);
  TestPoint[2].SetY(1.0);
  TestPoint[2].SetZ(0.0);
  TestPoint[3].SetX(0.0);
  TestPoint[3].SetY(1.0);
  TestPoint[3].SetZ(0.0);
  TestPoint[4].SetX(0.5);
  TestPoint[4].SetY(0.5);
  TestPoint[4].SetZ(0.0);

  // Make first vertex and first face
  psurf->MakeVertexFace(TestPoint[0], &VertPtr[0],
      &FacePtr[0]);

  // Make first edge and vertex
  psurf->MakeEdgeVertex(FacePtr[0], VertPtr[0],
      TestPoint[1], &EdgePtr[0], &VertPtr[1]);

  // Make second edge and vertex
  psurf->MakeEdgeVertex(FacePtr[0], VertPtr[1],
      TestPoint[2], &EdgePtr[1], &VertPtr[2]);

  // Make third edge and vertex
  psurf->MakeEdgeVertex(FacePtr[0], VertPtr[2],
      TestPoint[3], &EdgePtr[2], &VertPtr[3]);

  // Make fourth edge and second face
  psurf->MakeEdgeFace(VertPtr[0], VertPtr[3], FacePtr[0],
      &EdgePtr[3], &FacePtr[1]);

  // Make fifth edge and third face
  psurf->MakeEdgeFace(VertPtr[0], VertPtr[2], FacePtr[1],
      &EdgePtr[4], &FacePtr[2]);

  // Split fifth edge
  psurf->SplitEdgeMakeVertex(EdgePtr[4], TestPoint[4],
      &EdgePtr[5], &VertPtr[4]);

  DisplaySurfData(psurf);   // Display surface data

  // Join fifth and sixth edges
  psurf->JoinEdgeKillVertex(EdgePtr[4], EdgePtr[5],
      VertPtr[4]);

  // Kill third face and fifth edge
  psurf->KillEdgeFace(EdgePtr[4], FacePtr[2]);

  // Kill second face and fourth edge
  psurf->KillEdgeFace(EdgePtr[3], FacePtr[1]);

  // Kill third edge
  psurf->KillEdgeVertex(EdgePtr[2], VertPtr[3]);

  // Kill second edge
  psurf->KillEdgeVertex(EdgePtr[1], VertPtr[2]);

  // Kill first edge
  psurf->KillEdgeVertex(EdgePtr[0], VertPtr[1]);

  // Kill first vertex and first face
  psurf->KillVertexFace();
  
  delete psurf;     // Delete surface

  return 0;
}

// Display surface data
void DisplaySurfData( WSurf *ps )
{
  unsigned int cnt;     // Counter
  WEdge *pedge;         // Edge pointer
  WFace *pface;         // Face pointer
  WVert *pvert;         // Vertex pointer

  // Display edge list data
  puts("Edge Data\n---------");

  if ((pedge = ps->GetEdgeListPtr()) != NULL)
  {
    cnt = 1;    // Initialize counter

    do  // Walk edge ring
    {
      printf("Edge %05u\n", cnt++);
      DisplayEdgeData(pedge);
      pedge = pedge->GetNext();
    }
    while (pedge != ps->GetEdgeListPtr());
  }

  // Display face list data
  puts("\nFace Data\n---------");

  if ((pface = ps->GetFaceListPtr()) != NULL)
  {
    cnt = 1;    // Initialize counter

    do  // Walk face ring
    {
      printf("Face %05u\n", cnt++);
      DisplayFaceData(pface);
      pface = pface->GetNext();
    }
    while (pface != ps->GetFaceListPtr());
  }

  // Display vertex list data
  puts("\nVertex Data\n-----------");

  if ((pvert = ps->GetVertListPtr()) != NULL)
  {
    cnt = 1;    // Initialize counter

    do  // Walk vertex ring
    {
      printf("Vertex %05u\n", cnt++);
      DisplayVertData(pvert);
      pvert = pvert->GetNext();
    }
    while (pvert != ps->GetVertListPtr());
  }

  putchar('\n');
}

// Display edge data
void DisplayEdgeData( WEdge *pe )
{
  printf("  Address:            %p\n", pe);
  printf("  Next edge:          %p\n", pe->GetNext());
  printf("  Previous edge:      %p\n", pe->GetPrev());
  printf("  Left face:          %p\n", pe->GetLeftFacePtr());
  printf("  Right face:         %p\n", pe->GetRightFacePtr());
  printf("  Head vertex:        %p\n", pe->GetHeadVertexPtr());
  printf("    Left edge wing:   %p\n", pe->GetHeadLeftWingPtr());
  printf("    Right edge wing:  %p\n",
      pe->GetHeadRightWingPtr());
  printf("  Tail vertex:        %p\n", pe->GetTailVertexPtr());
  printf("    Left edge wing:   %p\n", pe->GetTailLeftWingPtr());
  printf("    Right edge wing:  %p\n",
      pe->GetTailRightWingPtr());
}

// Display face data
void DisplayFaceData( WFace *pf )
{
  WLink *pcl;   // Current edge link pointer

  printf("  Address:            %p\n", pf);
  printf("  Next face:          %p\n", pf->GetNext());
  printf("  Previous face:      %p\n", pf->GetPrev());
  printf("  Edge list:\n");

  pcl = pf->GetLinkListPtr();
  if (pcl != NULL)
  {
    for ( ; ; )     // Walk edge link ring
    {
      printf("                      %p", pcl->GetEdgePtr());
      if ((pcl = pcl->GetNext()) != pf->GetLinkListPtr())
        putchar('\n');
      else
        break;
    }
  }

  putchar('\n');
}

// Display vertex data
void DisplayVertData( WVert *pv )
{
  WLink *pcl;   // Current edge link pointer

  printf("  Address:            %p\n", pv);
  printf("  Next vertex:        %p\n", pv->GetNext());
  printf("  Previous vertex:    %p\n", pv->GetPrev());
  printf("  Position:\n");
  printf("    x-axis:           %f\n",
      pv->GetPosn().GetX());
  printf("    y-axis:           %f\n",
      pv->GetPosn().GetY());
  printf("    z-axis:           %f\n",
      pv->GetPosn().GetZ());
  printf("  Edge list:\n");

  pcl = pv->GetLinkListPtr();
  if (pcl != NULL)
  {
    for ( ; ; )     // Walk edge link ring
    {
      printf("                      %p", pcl->GetEdgePtr());
      if ((pcl = pcl->GetNext()) != pv->GetLinkListPtr())
        putchar('\n');
      else
        break;
    }
  }

  putchar('\n');
}

