/*
 *  Routines for setting up the parameters/buffers/display
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "freq.h"
#include "extern.h"
#include "display.h"

int fftlen=1024;        /* Number of points for FFT */
long SampleRate=44100L; /* A/D sampling rate */
double fudgefactor=1;   /* Correction factor for sample rate errors */
int logfreq=0;          /* Flag set to 1 for log-based frequency scale */ 
int logamp=0;           /* Flag set to 1 for log-based amplitude scale */
int windfunc=0;         /* Flag set to selected window function */
float ys=1.0;           /* Flag set for max of y-axis */
int log_base=6;             /* Flag set for base of log scale (default=6 = -60db) */
int logs=0;             /* Flag for max of log scale (default=0 = 0db) */
int gain=0;             /* Amount of db/octave gain */
int gain3db=0;          /* Flag indicating a 3db/octave scale factor gain */
int deriv=0;            /* Flag for doing differencing for 6db/octave gain */
long ref_freq=1000;     /* Reference frequency for n db/octave gains */

char *windfile=NULL;    /* Pointer to filename for saving window data */

char *audio_device=NULL; /* Name of DSP device in Linux */
char *mixer_device=NULL; /* Name of the mixer device in Linux */
char *output_device=NULL; /* Name of the device for output echoing */
 
int Soundcard=SC_DEF;   /* Soundcard number (as defined in freq.h) */
int sb_irq=SB_IRQ;      /* IRQ used by the Soundblaster card */
int sb_dma=SB_DMA;      /* DMA channel used by the Soundblaster card */
int sb_addr=SB_ADDR;    /* I/O address of the Soundblaster card */
float maxfreq;
double alpha;           /* Gaussian window parameter */

char *window_name[] = { "Hamming","Hanning","Blackman","Gaussian",
			"Welch","Parzen","Rectangular" };

struct rgb background = { 0,0,20 },
	   warn       = { 20,0,0 },
	   graph      = { 30,35,60 },
	   tick       = { 40,40,40 },
	   label      = { 50,20,45 },
	   border     = { 40,40,40 },
	   text       = { 55,55,25 },
	   darkhl     = { 20,20,20 },
           lighthl    = { 63,63,63 };

/* Center frequencies for equalizer-mode display */
double center_freq[]={ 4,5,6.3,8,
                       10,12.5,16,20,25,31.5,40,50,63,80,
                       100,125,160,200,250,315,400,500,630,800,
                       1000,1250,1600,2000,2500,3150,4000,5000,6300,8000,
                       10000,12500,16000,20000,25000};
#define EQUALIZER_BINS (sizeof(center_freq)/sizeof(center_freq[0]))

/* Soundcard function pointers */
void (*init_soundcard)(void)=NULL;
void (*reset_soundcard)(void)=NULL;
void (*halt_soundcard)(void)=NULL;
void (*cleanup_soundcard)(void)=NULL;
void (*recordblock)(void FAR *)=NULL;
void (*set_mixer)(int,int)=NULL;
int sample_size;        /* Bits per sample (8 or 16) */
int mixers;             /* Mixers available (1) or not (0) */

#ifdef DOS

#include <alloc.h>

void FAR *aligned_malloc(long len)
{
   void far *ptr;
   unsigned int far *orig_ptr;
   unsigned seg,orig_seg,orig_off;
   long lin_addr;

   /* Allocate double the required space */
   ptr=farmalloc(len*2+32);
   if(ptr!=NULL)
   {
      /* Compute a new pointer to the buffer such that the next "len" bytes */
      /* do not cross a page boundary, and the offset is zero */
      /* (as required by DMA transfers) */
      orig_off=FP_OFF(ptr);
      orig_seg=FP_SEG(ptr);
      /* reserve 4 bytes for the original pointer */
      lin_addr=(orig_seg*16L)+orig_off+4L;
      if((lin_addr&0xF0000L) != ((lin_addr+len-1)&0xF0000L))
	 lin_addr=(lin_addr+len-1)&0xF0000L;
      else
	 lin_addr=(lin_addr+15)&0xFFFF0L;

      seg=(unsigned int)(lin_addr/16);
      orig_ptr=(unsigned far *)MK_FP(seg-1,0x000C);
      orig_ptr[0]=orig_off;
      orig_ptr[1]=orig_seg;
      ptr=MK_FP(seg,0);
      /*
	 printf("Original: %04x:%04x, New: %04x:%04x, Linear: %05lx\n",
	 orig_seg,orig_off,FP_SEG(ptr),FP_OFF(ptr),lin_addr);
	 */
   }
   return ptr;
}

void aligned_free(void far *ptr)
{
   if(ptr!=NULL)
   {
      unsigned far *old_ptr=(unsigned far *)MK_FP(FP_SEG(ptr)-1,0x000c);
      /*
	 printf("Freeing: %04x:%04x, Ptr: %04x:%04x\n",
	 FP_SEG(old_ptr),FP_OFF(old_ptr),FP_SEG(ptr),FP_OFF(ptr));
	 */
      farfree(MK_FP(old_ptr[1],old_ptr[0]));
   }
}
#endif /* DOS */

/*
 *  Parse the ini file, if it exists
 */
void parse_ini_file(void)
{
   int i;
   FILE *fp;
   char buffer[100];
   
   if((fp=fopen(ini_file,"r"))!=NULL)
   {
      while(!feof(fp))
      {
	 fgets(buffer,sizeof(buffer),fp);
	 for(i=0;(buffer[i]!=0) && (buffer[i]!=':');i++)
	    buffer[i]=toupper(buffer[i]);

	 if(strncmp(buffer,"SOUNDCARD:",10)==0)
            Soundcard=parsecardname(buffer+10);
	 sscanf(buffer,"SAMPLE RATE:%ld",&SampleRate);
         sscanf(buffer,"RATE FUDGE FACTOR:%lf",&fudgefactor);
	 sscanf(buffer,"FFT LENGTH:%d",&fftlen);
	 sscanf(buffer,"WINDOW FUNCTION:%d",&windfunc);
         sscanf(buffer,"BAR DISPLAY: %d\n",&barmode);
	 sscanf(buffer,"LOG FREQ SCALE:%d",&logfreq);
	 sscanf(buffer,"LOG AMP SCALE:%d",&logamp);
	 sscanf(buffer,"BASE DB:%d",&log_base);
	 sscanf(buffer,"TOP DB:%d",&logs);
	 sscanf(buffer,"MAX AMP:%f",&ys);
	 sscanf(buffer,"DB/OCTAVE GAIN:%d",&gain);
	 sscanf(buffer,"REFERENCE FREQUENCY:%ld",&ref_freq);
	 sscanf(buffer,"BASE FREQUENCY:%f\n",&freq_base);
	 sscanf(buffer,"FREQUENCY FACTOR:%f\n",&freq_scalefactor);
	 sscanf(buffer,"DECAY MODE:%d",&decay_mode);
	 sscanf(buffer,"DECAY FACTOR:%f",&decay_factor);
	 sscanf(buffer,"AVERAGING COUNT:%d",&decay_count);
	 sscanf(buffer,"BACKGROUND COLOR:%d,%d,%d",&background.red,&background.green,&background.blue);
	 sscanf(buffer,"CLIPPING WARNING COLOR:%d,%d,%d",&warn.red,&warn.green,&warn.blue);
	 sscanf(buffer,"GRAPH COLOR:%d,%d,%d",&graph.red,&graph.green,&graph.blue);
	 sscanf(buffer,"TICK MARK COLOR:%d,%d,%d",&tick.red,&tick.green,&tick.blue);
	 sscanf(buffer,"AXIS LABEL COLOR:%d,%d,%d",&label.red,&label.green,&label.blue);
	 sscanf(buffer,"BORDER COLOR:%d,%d,%d",&border.red,&border.green,&border.blue);
	 sscanf(buffer,"TEXT COLOR:%d,%d,%d",&text.red,&text.green,&text.blue);
	 sscanf(buffer,"CURSOR UPPER COLOR:%d,%d,%d",&darkhl.red,&darkhl.green,&darkhl.blue);
	 sscanf(buffer,"CURSOR LOWER COLOR:%d,%d,%d",&lighthl.red,&lighthl.green,&lighthl.blue);

         #ifdef UNIX
	    if(strncmp(buffer,"AUDIO DEVICE:",13)==0)
	    {
	       int i=13;
	       while(buffer[i]==' ') i++;  /* Remove leading spaces */
	       audio_device=(char *)malloc(strlen(buffer+i)+1);
	       strcpy(audio_device,buffer+i);
	       for(i=0;i<strlen(audio_device);i++) /* Remove trailing junk */
	       {
		  if(   (audio_device[i]==0x0d)
                     || (audio_device[i]==0x0a)
		     || (audio_device[i]==0x20))
		     audio_device[i]=0;
	       }
	    }
         #endif /* UNIX */
         #ifdef LINUX_MIXER
	    if(strncmp(buffer,"MIXER DEVICE:",13)==0)
	    {
	       int i=13;
	       while(buffer[i]==' ') i++;  /* Remove leading spaces */
	       mixer_device=(char *)malloc(strlen(buffer+i)+1);
	       strcpy(mixer_device,buffer+i);
	       for(i=0;i<strlen(mixer_device);i++) /* Remove trailing junk */
	       {
		  if(   (mixer_device[i]==0x0d)
                     || (mixer_device[i]==0x0a)
		     || (mixer_device[i]==0x20))
		     mixer_device[i]=0;
	       }
	    }
         #endif /* UNIX */
	 if(strncmp(buffer,"OUTPUT DEVICE:",14)==0)
	 {
	    int i=14;
	    while(buffer[i]==' ') i++;  /* Remove leading spaces */
	    output_device=(char *)malloc(strlen(buffer+i)+1);
	    strcpy(output_device,buffer+i);
	    for(i=0;i<strlen(output_device);i++) /* Remove trailing junk */
	    {
	       if(   (output_device[i]==0x0d)
		  || (output_device[i]==0x0a)
		  || (output_device[i]==0x20))
		  output_device[i]=0;
	    }
	 }
      }
      fclose(fp);
   }
}

void setnormalpalette(void)
{
   draw_setpalette(0,background.red,background.green,background.blue);
   draw_setpalette(LABEL_COLOR,label.red,label.green,label.blue);
   draw_setpalette(BORDER_COLOR,border.red,border.green,border.blue);
   draw_setpalette(TEXT_COLOR,text.red,text.green,text.blue);
   draw_setpalette(GRAPH_COLOR,graph.red,graph.green,graph.blue);
   draw_setpalette(DARK_HIGHLIGHT,darkhl.red,darkhl.green,darkhl.blue);
   draw_setpalette(LIGHT_HIGHLIGHT,lighthl.red,lighthl.green,lighthl.blue);
}

void setbwpalette(void)
{
   draw_setpalette(0,0,0,0);
   draw_setpalette(LABEL_COLOR,63,63,63);
   draw_setpalette(BORDER_COLOR,63,63,63);
   draw_setpalette(TEXT_COLOR,63,63,63);
   draw_setpalette(GRAPH_COLOR,63,63,63);
   draw_setpalette(DARK_HIGHLIGHT,20,20,20);
   draw_setpalette(LIGHT_HIGHLIGHT,63,63,63);
}

/*
 * Parse the command line
 */
void parse_command(int argc,char *argv[],char *environ[])
{
   int i=0;

   while(i<argc)
   {
      if(argv[i][0]=='-')
      {
         switch(argv[i][1])
         {
            case 'C':   /* Select the sound card */
            case 'c':
                 Soundcard=parsecardname(argv[i]+2);
                 break;
      	      #ifdef UNIX
	         case 'D':   /* Set the DSP device string */
	         case 'd':
		      audio_device=malloc(strlen(argv[i])-1);
		      strcpy(audio_device,argv[i]+2);
	              break;
              #endif /* UNIX */
              #ifdef LINUX_MIXER
	         case 'X':   /* Set the Mixer device string */
	         case 'x':
		      mixer_device=malloc(strlen(argv[i])-1);
		      strcpy(mixer_device,argv[i]+2);
	              break;
              #endif /* LINUX_MIXER */
	         case 'O':   /* Set the Output device string */
	         case 'o':
		      output_device=malloc(strlen(argv[i])-1);
		      strcpy(output_device,argv[i]+2);
	              break;
	         case 'S':   /* Set the sampling rate (in Hz) */
	         case 's':
		      SampleRate=atol(&argv[i][2]);
		      break;
	         case 'F':   /* Set the FFT length */
	         case 'f':
		      fftlen=atoi(&argv[i][2]);
		      break;
	         case 'M':   /* Set the maximum value for linear display */
	         case 'm':
		      ys=atof(&argv[i][2]);
		      break;
	         case 'T':   /* Set the maximum value for log display */
	         case 't':
		      logs=atoi(&argv[i][2]);
                      break;
                 case 'B':   /* Set the minimum value for log display */
	         case 'b':
		      log_base=atoi(&argv[i][2]);
		      break;
	         case 'G':   /* Set the db/octave gain factor */
	         case 'g':
		      gain=atoi(&argv[i][2]);
                      break;
	         case 'R':   /* Set the frequency where the gain is 0dB */
	         case 'r':
		      ref_freq=atol(&argv[i][2]);
		      break;
	         case 'L':   /* Set logarithmic amplitude/frequency scale */
	         case 'l':
		      if((argv[i][2]=='A')||(argv[i][2]=='a'))
		         logamp=1;
                      else if((argv[i][2]=='F')||(argv[i][2]=='f'))
		         logfreq=1;
		      else if((argv[i][2]=='E')||(argv[i][2]=='e'))
			 logfreq=2;
		      else
		      {
		         printf("Ignoring unrecognized switch: %s\n",argv[i]);
                         puts("Hit <ENTER> to continue...");
                         getchar();
		      }
                      break;
                 case 'W':   /* Select the window function */
	         case 'w':
		      if(argv[i][2]>='A')
		      {
		         windfile=&argv[i][2];
		      }
                      else
		      {
		         windfunc=argv[i][2]-'0';
            	         if(windfunc==3)
		         {
			    alpha=0;
			    if(argv[i][3]==',')
			       alpha=atof(&argv[i][4]);
			    if(alpha<=0) alpha=1;
			 }
		      }
		      break;
	         case '?':   /* Display some help information */
                 case 'H':
	         case 'h':
              #ifndef UNIX
                 puts("-Cnumber selects the soundcard (0=SB, 1=PAS, 2=VESA, 3=SB16).");
              #else /* UNIX */
                 puts("-Cnumber selects the soundcard (0=AU, 1=DSP).");
		 puts("-Ddevice selects the DSP device.");
		 puts("-Xdevice selects the Mixer device.");
              #endif /* UNIX */
		 puts("-Odevice selects the output device.");
                 puts("-Snumber sets the sampling rate.");
                 puts("-Fnumber sets the length of the FFT.");
                 puts("-Mnumber sets the scale maximum.");
                 puts("-Bnumber sets the logarithmic base level (in tens of dB).");
                 puts("-Tnumber sets the logarithmic top level (in tens of dB).");
                 puts("-Gnumber sets the dB/octave gain factor.");
                 puts("-Rnumber sets the reference frequency for dB/octave gains.");
                 puts("-LA sets a logarithmic scale for the amplitude axis.");
                 puts("-LF sets a logarithmic scale for the frequency axis.");
		 puts("-LE sets an equalizer band display for the frequency axis.");
                 puts("-W0 selects a Hamming window.  (offset sine) <--default");
                 puts("-W1 selects a Hanning window.  (sine)");
                 puts("-W2 selects a Blackman window. (two sines)");
                 puts("-W3[,alpha] selects a Gaussian window.");
                 puts("-W4 selects a Welch window.    (quadratic)");
                 puts("-W5 selects a Parzen window.   (triangular)");
                 puts("-W6 selects a Rectangular window.");
                 puts("-Wfilename saves the windowing function to the specified file.");
                 puts("-? or -H displays this message.");
                 exit(0);
            default:
                 printf("Ignoring unrecognized switch: %s\n",argv[i]);
                 puts("Press <ENTER> to continue...");
                 getchar();
         }
      }
      else
         printf("Ignoring unrecognized parameter: %s  (use -h for help)\n",argv[i]);
      i++;
   }

   /*
    * Set up the flags based on chosen gain
    */
   if(gain>11)
   {
      gain=12;
      gain3db=0;
      deriv=2;
   }
   else if(gain>8)
   {
      gain=9;
      gain3db=1;
      deriv=1;
   }
   else if(gain>5)
   {
      gain=6;
      gain3db=0;
      deriv=1;
   }
   else if(gain>2)
   {
      gain=3;
      gain3db=1;
      deriv=0;
   }
   else
   {
      gain=0;
      gain3db=0;
      deriv=0;
   }
   /*
    * Watch for bad choice of log scales
    */
   if(log_base<=logs) logs=log_base-3;
   if(ys>2.0) ys=2.0;
   if(ys<0.01) ys=0.01;

   if(SampleRate>88200L) SampleRate=88200L;
   if(SampleRate<5000L) SampleRate=5000L;

   if(fftlen<8) fftlen=8;
   if(fftlen>MAX_LEN) fftlen=MAX_LEN;
   /* Convert fftlen to a power of 2 */
   for(i=0;fftlen>1;fftlen>>=1,i++);
   if(i) fftlen<<=i;

   if(decay_mode<0) decay_mode=0;
   if(decay_mode>3) decay_mode=0;
   if(decay_count<1) decay_count=1;
   if(decay_count>MAX_DECAY_COUNT) decay_count=MAX_DECAY_COUNT;
   if(decay_factor<0.00001) decay_factor=0.00001;
   if(decay_factor>0.99999) decay_factor=0.99999;

   /* Set up the soundcard function pointers */
#ifdef SC_SB8
   if(Soundcard==SC_SB8)
      init_sb8(environ);
#endif
#ifdef SC_PAS16
   if(Soundcard==SC_PAS16)
      init_pas16();
#endif
#ifdef SC_VESA
   if(Soundcard==SC_VESA)
      init_vesa();
#endif
#ifdef SC_SB16
   if(Soundcard==SC_SB16)
      init_sb16(environ);
#endif
#ifdef SC_LINUX
   if(Soundcard==SC_LINUX)
      init_linux_sc();
#endif
#ifdef SC_MULAW
   if(Soundcard==SC_MULAW)
      init_mulaw_sc();
#endif
   if(reset_soundcard==NULL)
   {
      puts("Error: Invalid soundcard selection");
      puts("Valid choices are:");
      #ifdef SC_SB8
         printf("  %s (%d)\n",SC_SB8_NAME,SC_SB8);
      #endif
      #ifdef SC_PAS16
         printf("  %s (%d)\n",SC_PAS16_NAME,SC_PAS16);
      #endif
      #ifdef SC_VESA
         printf("  %s (%d)\n",SC_VESA_NAME,SC_VESA);
      #endif
      #ifdef SC_SB16
         printf("  %s (%d)\n",SC_SB16_NAME,SC_SB16);
      #endif
      #ifdef SC_MULAW
         printf("  %s (%d)\n",SC_MULAW_NAME,SC_MULAW);
      #endif
      #ifdef SC_LINUX
         printf("  %s (%d)\n",SC_LINUX_NAME,SC_LINUX);
      #endif
      exit(1);
   }
}

/* Parse a sound card number or label */
int parsecardname(char *label)
{
   int i=0,j;
   char name[10];

   /* Skip over any leading space */
   while(label[i]==' ') i++;

   /* Check for numerical values */
   if(label[i]<='9')
      return(label[i]-'0');

   /* Convert the string to upper case */
   for(j=0;label[i] && (j<9);i++,j++)
      name[j]=toupper(label[i]);
   name[j]=0;

   /* Check for valid string values */
   #ifdef SC_SB8
      if(strncmp(name,SC_SB8_NAME,strlen(SC_SB8_NAME))==0)
         return SC_SB8;
   #endif
   #ifdef SC_PAS16
      if(strncmp(name,SC_PAS16_NAME,strlen(SC_PAS16_NAME))==0)
         return SC_PAS16;
   #endif
   #ifdef SC_VESA
      if(strncmp(name,SC_VESA_NAME,strlen(SC_VESA_NAME))==0)
         return SC_VESA;
   #endif
   #ifdef SC_SB16
      if(strncmp(name,SC_SB16_NAME,strlen(SC_SB16_NAME))==0)
         return SC_SB16;
   #endif
   #ifdef SC_MULAW
      if(strncmp(name,SC_MULAW_NAME,strlen(SC_MULAW_NAME))==0)
         return SC_MULAW;
   #endif
   #ifdef SC_LINUX
      if(strncmp(name,SC_LINUX_NAME,strlen(SC_LINUX_NAME))==0)
         return SC_LINUX;
   #endif
   return(-1); /* If none match, return failure */
}

/*
 *  Allocate memory for and initialize the buffers
 */
void setup_buffers(int length)
{
   int i;

   for(i=0;i<BUFFERS;i++)
   {
     #ifdef DOS
      if((buffer[i]=aligned_malloc(length*(sample_size/8)))==NULL)
      {
         puts("Unable to allocate enough memory for the buffers!");
         exit(1);
      }
      /* Pre-set the buffer to all zeros */
      _fmemset(buffer[i],0,length*(sample_size/8));
      flag[i]=0;

     #else /* DOS */

      if((buffer[i]=malloc(length*(sample_size/8)))==NULL)
      {
         puts("Unable to allocate enough memory for the buffers!");
         exit(1);
      }
      /* Pre-set the buffer to all zeros */
      memset(buffer[i],0,length*(sample_size/8));
      flag[i]=0;
     #endif /* DOS */
   }

   if((fftdata=(short *)malloc(length*sizeof(short)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((wind=(short *)malloc(length*sizeof(short)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((displayval=(long *)malloc(length/2*sizeof(long)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((lastval=(long *)malloc(length/2*sizeof(long)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((lastval_f=(float *)malloc(length/2*sizeof(float)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((ybase=(long *)malloc(length/2*sizeof(long)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if((databuf=(long FAR **)malloc(MAX_DECAY_COUNT*sizeof(long FAR *)))==NULL)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   for(max_decay_count=0;max_decay_count<MAX_DECAY_COUNT;max_decay_count++)
   {
      #ifdef DOS
         if((databuf[max_decay_count]=(long FAR *)farmalloc(length/2*sizeof(long)))==NULL)
            break;
      #else
         if((databuf[max_decay_count]=(long *)malloc(length/2*sizeof(long)))==NULL)
            break;
      #endif
   }
   if(max_decay_count<1)
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   if(decay_count>max_decay_count) decay_count=max_decay_count;

   /*
    *  Clear out the memory buffers
    */
   for(i=0;i<(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      lasty[i]=WINDOW_BOTTOM;

   reset_decay_buffers();
}

void compute_window_function(void)
{
   int i;
   /*
    *  Calculate FFT Windowing function
    */
   for(i=0;i<fftlen;i++)
   {
      double val;

      switch(windfunc)
      {
         case 1:    /* Hanning */
            val = 0.5 - 0.5*cos(2*M_PI*i/fftlen);
            break;

         case 2:    /* Blackman */
            val = 0.42-0.5*cos(2*M_PI*i/fftlen)+0.08*cos(4*M_PI*i/fftlen);
            break;

	 case 3:    /* Gaussian */
	    val = exp(-alpha / ((double)fftlen*(double)fftlen)
			     * ((double)2*i-fftlen)*((double)2*i-fftlen));
	    break;

	 case 4:    /* Welch */
	    val = 1 -  ((double)(2*i-fftlen)/(double)(fftlen+1))
		      *((double)(2*i-fftlen)/(double)(fftlen+1));
	    break;

	 case 5:    /* Parzen */
	    val = 1 - fabs((double)(2*i-fftlen)/(double)(fftlen+1));
	    break;

	 case 6:    /* Rectangular */
	    val = 1;
	    break;

	 default:   /* Hamming */
	    val = 0.54-0.46*cos(2*M_PI*i/fftlen);
	    break;
      }
      wind[i]=floor(val*32767+0.5);
   }

   /*
    *  If output requested, save the window function. (for the curious)
    */
   if(windfile)
   {
      FILE *fp;
      if((fp=fopen(windfile,"w"))==NULL)
      {
         printf("Error opening window output file: %s\n",windfile);
	 puts("Press <ENTER> to continue");
         getchar();
      }
      else
      {
         for(i=0;i<fftlen;i++)
            fprintf(fp,"%d\n",wind[i]);
         fclose(fp);
      }
      windfile=NULL;
   }
}


/*
 *  Do some range checking on the base and scale factors
 */
void xrange_check(void)
{
   if(freq_scalefactor<1.0) freq_scalefactor=1.0;
   if(freq_scalefactor>16.0) freq_scalefactor=16.0;

   if(logfreq)
   {
      double max_base=SampleRate/2/exp(log(fftlen/2)/freq_scalefactor);
      if(freq_base<(double)SampleRate/fftlen) freq_base=(double)SampleRate/fftlen;
      if(freq_base>max_base) freq_base=max_base;
   }
   else
   {
      if(freq_base<0) freq_base=0;
      if((freq_base+SampleRate/(2*freq_scalefactor))>SampleRate/2)
	 freq_base=SampleRate/2-SampleRate/(2*freq_scalefactor);
   }
   if(logfreq)
      maxfreq=freq_base*exp(log(fftlen/2)/freq_scalefactor);
   else
      maxfreq=freq_base+(double)SampleRate/(freq_scalefactor*2.0);
}

/*
 *  Set up X axis scales
 */
void setup_xscale(void)
{
   int i;

   /*
    *  Do some range checking on the base and scale factors
    */
   xrange_check();

   if(logfreq==2)
   {
      /* Do RMS averaging into a fixed set of bins */
      int j;
      double start_freq[EQUALIZER_BINS];

      start_freq[0]=0;
      i=0;
      for(j=1;j<EQUALIZER_BINS;j++)
	 start_freq[j]=sqrt( center_freq[j]*center_freq[j-1] );
      j=0;
      for(i=1;i<=(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      {
	 /* Generate frequencies on a logarithmic scale */
	 /* Note: bin#=exp( (i/width) * log(fftlen/2) ) */
         /* where i is the display line, and width is the display width */
	 double val=(double)SampleRate/(double)fftlen
                      *exp((double)i/(double)(WINDOW_RIGHT-WINDOW_LEFT)
                      *log(fftlen/2));
	 for(;(val>start_freq[j]) && (j<EQUALIZER_BINS-1);j++);
	 if(--j<0) j=0;
	 x[i]=floor(start_freq[j]/SampleRate*fftlen+0.99);
	 if(x[i]<0) x[i]=0;
	 if(x[i]>=fftlen/2) x[i]=fftlen/2-1;
      }
      x[0]=x[1];  /* Initialize the zero bin */
      
      /* Set up the count-to bins */
      i=WINDOW_RIGHT-WINDOW_LEFT;
      if(j<EQUALIZER_BINS-1) j++;
      x2[i]=floor(start_freq[j]/SampleRate*fftlen+0.99);
      if(x2[i]>=fftlen/2) x2[i]=fftlen/2-1;
      for(;i>0;i--)
      {
	 if(x[i]==x[i-1])
       	    x2[i-1]=x2[i];
	 else
	    x2[i-1]=x[i];
      }
   }
   else
   {
      /*
       *  Initialize graph x scale (linear or logarithmic).
       *  This array points to the bin to be plotted on a given line.
       */
      for(i=0;i<=(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      {
	 int val;
	 if(logfreq)
	    val=floor((double)fftlen*freq_base/(double)SampleRate
                      *exp((i-0.45)/(double)(WINDOW_RIGHT-WINDOW_LEFT)
                      *log(fftlen/2)/freq_scalefactor)+0.5);
	 else
	    val=floor(((i-0.45)/(double)(WINDOW_RIGHT-WINDOW_LEFT)
                      *(double)fftlen/2.0/freq_scalefactor)
                      +(freq_base/(double)SampleRate*(double)fftlen)+0.5);
	 
	 if(val<0) val=0;
	 if(val>=fftlen/2) val=fftlen/2-1;
	 
	 if(i>0)
	    x2[i-1]=val;
	 if(i<=(WINDOW_RIGHT-WINDOW_LEFT))
	    x[i]=val;
      }
      /* Compute the ending locations for lines holding multiple bins */
      for(i=0;i<=(WINDOW_RIGHT-WINDOW_LEFT);i++)
      {
	 if(x2[i]<=(x[i]+1))
	    x2[i]=0;
      }
   }
   
   /*
    *  If lines are repeated on the screen, flag this so that we don't
    *  have to recompute the y values.
    */
   for(i=(WINDOW_RIGHT-WINDOW_LEFT);i>0;i--)
   {
      if(x[i]==x[i-1])
      {
	 x[i]=-1;
	 x2[i]=0;
      }
   }
   
   if(logamp)
      setup_logscales();
   else
      setup_linscales();
   frequency_scale();
}

/*
 *  Set up logarithmic amplitude scale factors and offsets.
 */
void setup_logscales(void)
{
   int i;
   double scale,base,convert,offset;

   /*
    *  Compute the (logarithmic) y scale factor and offset.
    *  This may include a 3dB/octave gain.
    *
    *  Conversion factor from db/10 to dPhils (the computed "unit")
    *  where a factor of 2 yields 16384 dPhils (6.02dB)
    *  Scaling factor is such that  32768 ->   0.00 dB -> 245760 dPhils
    *                          and      2 -> -84.29 dB ->  16384 dPhils
    *                          and      1 -> -90.31 dB ->      0 dPhils
    *  i.e. dPhils=16384.0/log(2) * log(value)
    * and changes of 6.02 dB = 16384 dPhils
    */
   convert=819.2*log(10)/log(2);  /* Scale for dB to dPhils conversion */
   offset=log10(32768L)*20;       /* Offset for db to dPhils conversion */

   /*
    * This value is used in the main program group to convert squared values
    * amplitudes to dPhils using dPhils = log(value^2)*log_scalefactor
    */
   log_scalefactor=8192.0/log(2);

   scale=(WINDOW_BOTTOM-WINDOW_TOP)/(10*(log_base-logs)*convert);
   if(deriv==0)
      base=(offset-log_base*10)*convert;
   else if(deriv==1)
      base=(offset-log10(SampleRate/(2*M_PI*ref_freq))*20-log_base*10)*convert;
   else
      base=(offset-log10(SampleRate/(2*M_PI*ref_freq))*40-log_base*10)*convert;

   disp_scalefactor=scale;  /* Save the unshifted version for avg. display mode */

   shift=0;
   /*
    *  Make maximum use of available bits
    *  (use only 12 bits--other 4 used for higher resolution in the data)
    */
   while(scale<4096)
   {
      scale*=2;
      shift++;
   }
   scale=floor(scale+0.5);

   for(i=0;i<(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      yscale[i]=(unsigned int)scale;

   if(gain3db)
   {
      ybase[0]=500000L;
      for(i=1;i<fftlen/2;i++)
	 ybase[i]=(long)floor(0.5+base - log10((double)i*(double)SampleRate
                        /(double)fftlen/(double)ref_freq)*convert*10);
   }
   else
   {
      for(i=0;i<fftlen/2;i++)
	 ybase[i]=(long)floor(0.5+base);
   }
   shiftscale=1/pow(2,shift);
}

/*
 *  Set up linear amplitude scale factors
 */
void setup_linscales(void)
{
   int i;
   double scale;

   /*
    *  Compute the (linear) y scale factor.
    *  This may include a 3dB/octave gain.
    */
   scale=(WINDOW_BOTTOM-WINDOW_TOP)/(ys*32768.0*sqrt(ref_freq));

   shift=4; /* Display data has an extra factor of 16 for better resolution */

   if(deriv==1)
   {
      scale*=(double)SampleRate/(2*M_PI*(double)ref_freq);
   }
   else if(deriv==2)
   {
      scale*=(double)SampleRate*(double)SampleRate
	     /(4*M_PI*M_PI*(double)ref_freq*(double)ref_freq);
   }

   /*
    *  Make maximum use of available bits
    */
   if(gain3db)
   {
      /*
       *  Make maximum use of available bits 
       *  (use only 12 bits--other 4 used for higher resolution in the data)
       */
      while((scale*sqrt(SampleRate/2))<4096)
      {
	 scale*=2;
	 shift++;
      }
      for(i=0;i<(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      {
	 if(x[i]==-1)
	    yscale[i]=0;
	 else
	    yscale[i]=floor(0.5+scale*sqrt((double)x[i]*(double)SampleRate/(double)fftlen));
      }
   }
   else
   {
      /*
       *  Make maximum use of available bits 
       *  (use only 12 bits--other 4 used for higher resolution in the data)
       */
      scale=scale*sqrt(ref_freq);
      while(scale<4096)
      {
	 scale*=2;
	 shift++;
      }

      scale=floor(scale+0.5);

      for(i=0;i<(WINDOW_RIGHT-WINDOW_LEFT+1);i++)
      {
	 if(x[i]==-1)
	    yscale[i]=0;
	 else
	    yscale[i]=scale;
      }
   }
   shiftscale=1/pow(2,shift);
}

void frequency_scale(void)
{
   int i;

   double step,freq;
   char text[20];
   /*
    *  Put up the frequency scale.
    */
   draw_bar(WINDOW_LEFT-10,WINDOW_BOTTOM+3,WINDOW_RIGHT+10,469,0);

   draw_fontcolor(LABEL_COLOR);

   if(logfreq==2)
   {
      /* Put up the equalizer band values */

      for(i=0;i<EQUALIZER_BINS;i++)
      {
	 int x;
	 double freq=center_freq[i];
	 x=floor(log(freq/SampleRate*fftlen)/log(fftlen/2)
                 *(double)(WINDOW_RIGHT-WINDOW_LEFT)+0.5)+WINDOW_LEFT;
	 if((x>=WINDOW_LEFT) && (x<=WINDOW_RIGHT))
	 {
	    int y=WINDOW_BOTTOM+3;
	    draw_line(x,y,x,y+3,BORDER_COLOR);
	    sprintf(text,"%.0f",freq);
	    draw_text_vertical(x-_font_width/2,y+6,text);
	 }
      }
  }
   else
   {
      if(logfreq)
	 step=log(fftlen/2)/(32.0*freq_scalefactor);
      else
	 step=(double)SampleRate/(64.0*freq_scalefactor);
      
      for(i=0;i<=32;i++)
      {
	 int x=WINDOW_LEFT+((double)i*(WINDOW_RIGHT-WINDOW_LEFT)/32.0);
	 int y=WINDOW_BOTTOM+3;
	 draw_line(x,y,x,y+3,BORDER_COLOR);
	 if(logfreq)
	    freq=freq_base*exp(step*i);
	 else
	    freq=freq_base+i*step;
	 if(freq<0) freq=0;
	 sprintf(text,"%.0f",freq);
	 draw_text_vertical(x-_font_width/2,y+6,text);
      }
   }

   draw_fontcolor(TEXT_COLOR);
}

void amplitude_scale(void)
{
   int i;
   char text[20];

   draw_bar(0,WINDOW_TOP-10,WINDOW_LEFT-3,WINDOW_BOTTOM+7,0);
   draw_bar(WINDOW_RIGHT+3,WINDOW_TOP-10,639,WINDOW_BOTTOM+7,0);

   draw_fontcolor(LABEL_COLOR);

   /*
    *  Put up the amplitude scale
    */
   if(logamp)
   {
      for(i=0;i<(log_base-logs)+1;i++)
      {
         int x=WINDOW_LEFT-3;
         int y=WINDOW_TOP+i*(WINDOW_BOTTOM-WINDOW_TOP)/(log_base-logs);
         draw_line(x,y,x-3,y,BORDER_COLOR);
	 sprintf(text,"%d",(i+logs)*(-10));
         draw_text_right(x-6,y-_font_height/2,text);
         draw_line(WINDOW_RIGHT+3,y,WINDOW_RIGHT+6,y,BORDER_COLOR);
         draw_text_left(WINDOW_RIGHT+9,y-_font_height/2,text);
      }
   }
   else
   {
      double scale=(double)(ys*32768.0)/(double)(WINDOW_BOTTOM-WINDOW_TOP);
      for(i=0;i<=10;i++)
      {
         int x=WINDOW_LEFT-3;
         int y=WINDOW_BOTTOM-i*ys*3276.8/scale;
	 draw_line(x,y,x-3,y,BORDER_COLOR);
	 if(ys>0.095)
	   sprintf(text,"%4.2f",(float)i*ys*0.1);
	 else
	    sprintf(text,"%5.3f",(float)i*ys*0.1);
         draw_text_right(x-6,y-_font_height/2,text);
         draw_line(WINDOW_RIGHT+3,y,WINDOW_RIGHT+6,y,BORDER_COLOR);
         draw_text_left(WINDOW_RIGHT+9,y-_font_height/2,text);
      }
   }
   draw_fontcolor(TEXT_COLOR);
}

void update_header(void)
{
   char ach[100];

   draw_bar(0,0,639,WINDOW_TOP-10,0);
   draw_bar(WINDOW_LEFT-5,WINDOW_TOP-10,WINDOW_RIGHT+5,WINDOW_TOP-4,0);

   sprintf(ach,"Sampling rate: %ld Hz",SampleRate);
   draw_text_centered(SRX,SRY,ach);
   sprintf(ach,"FFT Length: %d points",fftlen);
   draw_text_centered(FLX,FLY,ach);
   sprintf(ach,"Frequency resolution: %.4g Hz",(double)SampleRate/(double)fftlen);
   draw_text_centered(FRX,FRY,ach);
   sprintf(ach,"%s window function",window_name[windfunc]);
   draw_text_centered(WFX,WFY,ach);
   sprintf(ach,"Gain of %d dB per octave (ref: %ld Hz)",gain,ref_freq);
   draw_text_centered(DGX,DGY,ach);

   if(decay_mode==0)
      sprintf(ach,"No averaging");
   else if(decay_mode==1)
      sprintf(ach,"Step up/decay averaging (%g)",decay_factor);
   else if(decay_mode==2)
      sprintf(ach,"Exponential averaging (%g)",decay_factor);
   else
      sprintf(ach,"Uniform averaging (%2d/%2d)",decay_count_actual,decay_count);
   draw_text_centered(DCX,DCY,ach);

   if(mixers)
   {
      sprintf(ach,"Mic:%3d  Ext:%3d  CD:%3d",mic_level,ext_level,int_level);
      draw_text_left(LVX,LVY,ach);
   }
}

void show_help(void)
{
   draw_bar(0,0,639,WINDOW_TOP-10,0);
   draw_bar(WINDOW_LEFT-5,WINDOW_TOP-10,WINDOW_RIGHT+5,WINDOW_TOP-4,0);

   if(help_mode==1)
   {
      draw_text_left(20,2, "    'X' Toggle log/linear frequency");
      draw_text_left(20,14,"    'Y' Toggle log/linear amplitude");
      draw_text_left(20,26,"    'P' Toggle peak display mode");
      draw_text_left(20,38,"    'G' Select dB/octave gain");
      draw_text_left(20,50,"    'A' Select averaging mode");
      draw_text_left(20,62,"'+'/'-' Inc./dec. averaging");
      draw_text_left(20,74,"<SPACE> Freeze display");

      draw_text_left(340,2, "    'W' Switch windowing function");
      draw_text_left(340,14,"    'R' Change the sampling rate");
      draw_text_left(340,26,"    'F' Change the FFT length");
      draw_text_left(340,38,"    'L' Toggle logging of FFT data");
      draw_text_left(340,50,"    'S' Save state to INI file");
      draw_text_left(340,62,"'H','?' Toggle this help screen");
      draw_text_left(340,74,"'Q','E' Exit from the program");
   }
   else
   {
      draw_text_left(100,2, "<Left/Right> Shift the frequency axis");
      draw_text_left(100,14,"     '<','>' Expand/contract the frequency axis");
      draw_text_left(100,26,"   <Up/Down> Change the amplitude scale");
      draw_text_left(100,38," <PgUp/PgDn> Change the base level for log-amplitude");
      draw_text_left(100,50,"<Home>/<End> Move to the minimum/maximum freqency");
      if(mixers)
      {
         draw_text_left(36,62,"'B' Toggle bar/dot display");
         draw_text_left(36,74,"'C' Toggle B&W/color display");
         draw_text_left(340,62,"'V' Refresh display");
         draw_text_left(300,74,"(),[],{} Adjust mic,ext,CD input level");
      }
      else
      {
         draw_text_left(80,62,"'V' Refresh display     'B' Toggle B&W/color display");
      }
   }
}
