// ANIMSAMP.CPP: Simple animation demo using WinG. Tested with
// Microsoft VC++ 2.0 running under Windows NT. Always draws to an
// 8-bpp DIB, but the results can be displayed at any color depth.
// Adapted from the Cube sample program in the WinG SDK.

#include <windows.h>
#include <windowsx.h>
#include <wing.h>
#include "animsamp.hpp"

#define DRAW_DIRECT     1   // 0 to draw boxes via GDI, 1 for direct
#define MIN_DIB_WIDTH   100 // minimum dimensions of WinGBitmap,
#define MIN_DIB_HEIGHT  100 //  to avoid out-of-bounds drawing
// A box is the only sort of object in the world we'll draw
#define NUM_BOXES   7
typedef struct _box {
    int X;
    int Y;
    int Width;
    int Height;
    int XInc;
    int YInc;
} box;
box Boxes[NUM_BOXES] = {
    {0, 0, 20, 20, 5, 5},
    {0, 0, 30, 10, 4, 7},
    {0, 0, 60, 24, 5, 9},
    {0, 0, 24, 24, 7, 7},
    {0, 0, 16, 30, 8, 1},
    {0, 0, 40, 20, 2, 6},
    {0, 0, 24, 30, 4, 2},
};
static char szAppName[]="Sample WinG animation demo";
static int      appActive;
static HWND     hwndApp;
static HPALETTE hpalApp = 0;
static HDC      hdcWinG;
static HBITMAP  hbmOld;
static HBITMAP  hbmWinG;
struct {
    BITMAPINFOHEADER    Header;
    RGBQUAD             aColorTable[256];
} HeaderAndPalette = {
    sizeof(BITMAPINFOHEADER), 50, 50, 1, 8, BI_RGB, 0, 0, 0, 0, 0
};
static int DibWidth, DibHeight, DibPitch;;
char *pBits;

LRESULT CALLBACK AppWndProc(HWND hwnd, UINT msg, WPARAM wParam,
    LPARAM lParam);
void AppExit(void);
int AppIdle(void);
void AppPaint(HWND hwnd, HDC hdc);
void ClearSystemPalette(void);

// Load time initialization.

int LoadInit(HINSTANCE hInst,HINSTANCE hPrev,int sw,LPSTR szCmdLine)
{
    WNDCLASS cls;

    ClearSystemPalette();   // Make sure we can get the whole palette

    if (!hPrev) {
        cls.hCursor        = LoadCursor(0, IDC_ARROW);
        cls.hIcon          = 0;
        cls.lpszMenuName   = "AppMenu";
        cls.lpszClassName  = szAppName;
        cls.hbrBackground  = (HBRUSH)GetStockObject(BLACK_BRUSH);
        cls.hInstance      = hInst;
        cls.style          = CS_BYTEALIGNCLIENT | CS_VREDRAW |
                             CS_HREDRAW;
        cls.lpfnWndProc    = (WNDPROC)AppWndProc;
        cls.cbClsExtra     = 0;
        cls.cbWndExtra     = 0;
        if (!RegisterClass(&cls))
            return FALSE;
    }

    hwndApp = CreateWindow (szAppName, szAppName,
            WS_OVERLAPPEDWINDOW,
            CW_USEDEFAULT, 0, 350,350, 0, 0, hInst, 0);
    hdcWinG = WinGCreateDC();
    ShowWindow(hwndApp, sw);
    return TRUE;
}

// Main proc and message pump.

int CALLBACK WinMain(HINSTANCE hInst,HINSTANCE hPrev,LPSTR szCmdLine,
    int sw)
{
    MSG msg;

    if (!LoadInit(hInst, hPrev, sw, szCmdLine)) // initialize the app
        return FALSE;

    // Pump messages until quitting time, letting the idle proc
    // draw, if possible, when there's nothing else to do.
    for (;;) {
        if (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
            if (msg.message == WM_QUIT)
                break;
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        } else {
            if (AppIdle())
                WaitMessage();
        }
    }
    return msg.wParam;
}

// Idle loop; draws if the app is the active app or is an icon,
// does nothing otherwise.

int AppIdle()
{
    if (appActive || IsIconic(hwndApp)) {
        // Move all the objects
        for (int i=0; i<NUM_BOXES; i++) {
            // Bounce if at edge
            if (((Boxes[i].XInc < 0) &&
                ((Boxes[i].X + Boxes[i].XInc) < 0)) ||
                ((Boxes[i].XInc > 0) &&
                 ((Boxes[i].X + Boxes[i].Width + Boxes[i].XInc)
                 >= DibWidth))) {
                Boxes[i].XInc = -Boxes[i].XInc;
            }
            if (((Boxes[i].YInc < 0) &&
                 ((Boxes[i].Y + Boxes[i].YInc) < 0)) ||
                ((Boxes[i].YInc > 0) &&
                 ((Boxes[i].Y + Boxes[i].Height + Boxes[i].YInc)
                 >= DibHeight))) {
                Boxes[i].YInc = -Boxes[i].YInc;
            }
            Boxes[i].X += Boxes[i].XInc;
            Boxes[i].Y += Boxes[i].YInc;
        }
        // Draw the world with the new positions
        HDC hdc = GetDC(hwndApp);
        if (hpalApp) {
            SelectPalette(hdc, hpalApp, FALSE);
            RealizePalette(hdc);
        }
        AppPaint(hwndApp, hdc); // draw the world
        ReleaseDC(hwndApp, hdc);
        return FALSE;
    } else {
        return TRUE;    // nothing to do
    }
}

// Draws the current state of the world to the DIB (WinGBitmap),
// then copies the result to the passed-in DC (the screen).

void AppPaint(HWND hwnd, HDC hdc)
{
    // Clear the DIB to black
    PatBlt(hdcWinG, 0, 0, DibWidth, DibHeight, BLACKNESS);
#if DRAW_DIRECT
    GdiFlush(); // make sure this gets drawn right away, so it
                //  happens before the direct drawing (GDI batches
                //  drawing calls under Windows NT)
#endif

    // Draw the world (all the boxes) to the DIB
    for (int i=0; i<NUM_BOXES; i++) {
#if DRAW_DIRECT
        int Color = GetNearestPaletteIndex(hpalApp,
           RGB(((i+1)&0x04)*63, ((i+1)&0x02)*127, ((i+1)&0x01)*255));
        char *pTemp = pBits + (Boxes[i].Y * DibPitch) + Boxes[i].X;
        int Step = DibPitch - Boxes[i].Width;
        for (int j=0; j<Boxes[i].Height; j++) {
            for (int k=0; k<Boxes[i].Width; k++) {
                *pTemp++ = Color;
            }
            pTemp += Step;
        }
#else
        HBRUSH hbr;
        RECT rect;
        hbr = CreateSolidBrush(RGB(((i+1)&0x04)*63, ((i+1)&0x02)*127,
                ((i+1)&0x01)*255));
        rect.top = Boxes[i].Y;
        rect.left = Boxes[i].X;
        rect.bottom = Boxes[i].Y + Boxes[i].Height;
        rect.right = Boxes[i].X + Boxes[i].Width;
        FillRect(hdcWinG, &rect, hbr);
        DeleteObject(hbr);
#endif
    }

    // Copy the DIB to the screen.
    RECT rc;
    GetClientRect(hwndApp, &rc);
    if (IsIconic(hwndApp)) {
        WinGStretchBlt(hdc, 0, 0, rc.right, rc.bottom, hdcWinG, 0, 0,
                DibWidth, DibHeight);
    } else {
        WinGBitBlt(hdc, 0, 0, rc.right, rc.bottom, hdcWinG, 0, 0);
    }
    GdiFlush(); // make sure this gets drawn right away
}

// Main window proc. Receives all messages.

LRESULT CALLBACK AppWndProc(HWND hwnd, UINT msg, WPARAM wParam,
    LPARAM lParam)
{
    PAINTSTRUCT ps;
    HDC hdc;
    int f;
    int counter;

    switch (msg) {
        case WM_CREATE: 
            // Use the WinG halftone palette
            hpalApp = WinGCreateHalftonePalette();
            GetPaletteEntries(hpalApp, 0, 256,
                (PALETTEENTRY *)HeaderAndPalette.aColorTable);

            for(counter = 0; counter < 256; counter++) {
                // PALETTEENTRYs and RGBQUADs are reversed
                BYTE Temp =
                       HeaderAndPalette.aColorTable[counter].rgbBlue;
                HeaderAndPalette.aColorTable[counter].rgbBlue =
                        HeaderAndPalette.aColorTable[counter].rgbRed;
                HeaderAndPalette.aColorTable[counter].rgbRed = Temp;
            }
            break;

        case WM_ACTIVATEAPP:            // track if app in foreground
            appActive = (int)wParam;
            break;
            
        case WM_COMMAND:
            switch(wParam) {
                case MENU_EXIT:
                    PostMessage(hwnd, WM_CLOSE, 0, 0L);
                    break;
            }
            return 0L;

        case WM_DESTROY:                // clean up before leaving
            if (hpalApp)
                DeleteObject(hpalApp);
            if (hdcWinG) {
                SelectObject(hdcWinG, hbmOld);
                DeleteObject(hbmWinG);
                DeleteDC(hdcWinG);
            }
            PostQuitMessage(0);
            break;

        case WM_PALETTECHANGED:
            if ((HWND)wParam == hwnd)
                break;
            // if not current window doing the changing, fall through

        case WM_QUERYNEWPALETTE:
            hdc = GetDC(hwnd);
            if (hpalApp)
                SelectPalette(hdc, hpalApp, FALSE);
            f = RealizePalette(hdc);
            ReleaseDC(hwnd, hdc);
            if (f)      // if we got a realization, force a redraw
                InvalidateRect(hwnd, 0, FALSE);
            return f;

        case WM_PAINT:
            hdc = BeginPaint(hwnd, &ps);
            if (hpalApp) {
                SelectPalette(hdc, hpalApp, FALSE);
                RealizePalette(hdc);
            }
            AppPaint (hwnd, hdc);
            EndPaint(hwnd,&ps);
            return 0L;

        case WM_SIZE:
            if (wParam != SIZE_MINIMIZED) {
                // Create a WinGBitmap to match the client area
                if (hbmWinG) {
                    SelectObject(hdcWinG, hbmOld);
                    DeleteObject(hbmWinG);
                }
                RECT rect;
                GetClientRect(hwnd, &rect);

                // Set up the header for the WinGBitmap, making sure
                // it never gets so small that objects could draw
                // out-of-bounds
                WinGRecommendDIBFormat((BITMAPINFO *)
                        &HeaderAndPalette);
                DibWidth = (rect.right > MIN_DIB_WIDTH) ?
                        rect.right : MIN_DIB_WIDTH;
                DibPitch = (DibWidth+3) & ~0x03; // round up to dword
                DibHeight = (rect.bottom > MIN_DIB_HEIGHT) ?
                        rect.bottom : MIN_DIB_HEIGHT;
                HeaderAndPalette.Header.biWidth = DibWidth;
                HeaderAndPalette.Header.biHeight *= DibHeight;
                hbmWinG = WinGCreateBitmap(hdcWinG,
                        (BITMAPINFO *)&HeaderAndPalette,
                        (void **)&pBits);
                hbmOld = SelectBitmap(hdcWinG, hbmWinG);

                // If bottom-up bitmap, point to the top scan & make
                // the scan size negative to scan from top to bottom
                if (HeaderAndPalette.Header.biHeight > 0) {
                    pBits += (HeaderAndPalette.Header.biHeight - 1) *
                            DibPitch;
                    DibPitch = -DibPitch;
                }

                // Reset all the objects to the upper left to 
                // sure they're in the DIB
                for (int i=0; i<NUM_BOXES; i++)
                    Boxes[i].X = Boxes[i].Y = 0;
            }
    }
    return DefWindowProc(hwnd,msg,wParam,lParam);
}

// Fills and empties the system palette, so if this is the
// foreground app, it can be sure of grabbing all the non-static
// entries starting at entry #10.

void ClearSystemPalette(void)
{
    static struct {
        WORD Version;
        WORD NumberOfEntries;
        PALETTEENTRY aEntries[256];
    } Palette = {
        0x300,
        256
    };
  
    // Make the whole palette black, with nocollapse to force a
    // separate system palette entry for each black entry. The RGB
    // entries are statically initialized to zero
    for(int counter = 0; counter < 256; counter++)
        Palette.aEntries[counter].peFlags = PC_NOCOLLAPSE;

    // Realize the palette & discard it, to clear the system palette.
    HDC ScreenDC = GetDC(NULL);
    HPALETTE hpalScreen= CreatePalette((LOGPALETTE *)&Palette);
    if (hpalScreen) {
        hpalScreen = SelectPalette(ScreenDC, hpalScreen, FALSE);
        RealizePalette(ScreenDC);
        hpalScreen = SelectPalette(ScreenDC, hpalScreen, FALSE);
        DeleteObject(hpalScreen);
    }
    ReleaseDC(NULL, ScreenDC);
    return;
}
