/*
CHGDIR.C
Win32 Console app
Demonstrates that Win32 tasks in Win95 keep current drv/dir in Win16 TDB

Andrew Schulman
Senior editor, O'Reilly & Associates (Sebastopol CA)
andrew@ora.com

August 1995

This version is corrected from the version that appears in my book
*Unauthorized Windows 95* (IDG Books, 1994), pp. 446-448:

After the book was written, the GetCurrentThreadId() function in
Win95 was changed to return obfuscated thread IDs.  CHGDIR.C now uses
the TIDToTDB() function from UNOBFUSC.C to de-obfuscate the thread
ID into a Ring 3 thread control block.

Also after the book was written, KERNEL32.DLL stopped exporting the
VxDCall() function by name.  This function has ordinal #1 in KERNEL32.DLL.
However, KERNEL32 does not allow imports by ordinal.  To access VxDCall,
CHGDIR.C now uses the GetK32ProcAddress() function from K32EXP.C.

cl chgdir.c unobfusc.c k32exp.c

For instructions on using this program, see *Unauthorized Windows
95*, p. 450.
*/

#include <stdlib.h>
#include <stdio.h>
#define  WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "unobfusc.h"
#include "k32exp.h"

DWORD (WINAPI *VxDCall)(DWORD srvc, DWORD eax, DWORD ecx);

#define GET_PROC(mod, func) GetProcAddress(GetModuleHandle(mod), (func))
#define VWIN32_INT21_CALL   0x2A0010
#define VWIN32_INT31_CALL   0x2A0029
#define DosCall(eax, ecx)   VxDCall(VWIN32_INT21_CALL, (eax), (ecx))
#define DPMICall(eax, ecx)  VxDCall(VWIN32_INT31_CALL, (eax), (ecx))

char curdir[MAX_PATH], buf[MAX_PATH], bufupr[MAX_PATH];

void fail(const char *s) { puts(s); exit(1); }

main()
{
    HANDLE id;
    BYTE *tdb_drv, *tdb_curdir, *tdb_base;
    WORD tdb, psp;
    
    // Thread Id == Ring 3 THCB (Thread Control Block)
    id = GetCurrentThreadId();
    printf("Thread ID: %08lXh\n", id);
    
    // thread id unobfuscator from UNOBFUSC.C
    if ((id = TIDToTDB(id)) == 0)
        fail("Couldn't find thread ID unobfuscator XOR pattern");
    
    printf("TDB: %08lX\n", id);
        
    // Win16 TDB is at offset 1Ch in Ring 3 THCB
    // I'm a little surprised it's so easy to get at this!
    tdb = *((WORD *) (((BYTE *) id) + 0x1c));
    printf("TDB: %04Xh\n", tdb);
    
    // Use run-time dynamic linking to turn strings
    // into callable function pointers 
#if 1
    if ((VxDCall = GetK32ProcAddress(1)))
#else
    if ((VxDCall = GET_PROC("KERNEL32", "VxDCall0")))       // redo!!
#endif      
    {
        // Get linear base address for TDB by calling
        // DPMI INT 31h function 6 (Get Selector Base).
        // Even though we're a 32-bit app, we get the 16-bit
        // DPMI services, since we're calling indirectly.
        _asm mov bx, tdb
        DPMICall(0x0006, 0);
        _asm mov word ptr tdb_base+2, cx
        _asm mov word ptr tdb_base, dx
        printf("TDB base: %08lXh\n", tdb_base);
        
        // Call DOS INT 21h function 62h (Get PSP)
        DosCall(0x6200, 0);
        _asm mov psp, bx
        printf("PSP: %04Xh\n\n", psp);

        // Sanity check:  is TDB[60h] == PSP?
        if (*((WORD *) (tdb_base + 0x60)) != psp)
            fail("TDB and PSP don't match!");
            
        // Current drv at offset 66h in TDB, curr dir at offset 100h
        // SetCurrentDirectory changes these.
        // Note: Current drv/dir on per-task basis, not per-thread.
        tdb_curdir = tdb_base + 0x100;
        tdb_drv = tdb_base + 0x66;
    }
    else
        printf("Warning: Can't access VxDCall0\n\n");
    
    for (;;)
    {
        if (GetCurrentDirectory(MAX_PATH, curdir) == 0)
            printf("invalid>");
        else
            printf("[%s] ", curdir);
        gets(buf);
        if ((buf[0] == '\0') || (buf[0] == '\n')) 
            continue;
        strcpy(bufupr, buf);
        strupr(bufupr);
        #define MATCH(cmd)  (strncmp(bufupr, (cmd), sizeof(cmd)-1) == 0)
        if (MATCH("EXIT"))
            break;
        else if (MATCH("CD "))
        {
            if (! SetCurrentDirectory(&buf[3]))
                printf("CD failed\n");
            else if (tdb_curdir)
            {
                printf("TDB+66h:  %02Xh\n", *tdb_drv);
                printf("TDB+100h: \'%s\'\n", tdb_curdir);
            }
        }
        else if (MATCH("MD "))
        {
            if (! CreateDirectory(&buf[3], 0))
                printf("MD failed\n");
        }
        else
            WinExec(buf, SW_NORMAL);
    }
    printf("done\n");
    return 0;
}
