///////////////////////////////////////
// File:      gcmodes.cpp
// Libraries: ddraw.lib
// Author:    Michael J. Norton
// Date:      082396
//
// pixelDyne copyright 1994,1995,1996
// Developer:  Date:     Comments 
// mjn         082396    game console alpha
// mjn         092396    flip mod for sysmem
///////////////////////////////////////
#define WIN32_LEAN_AND_MEAN
#define GCMODES_CPP_REV "gcmodes.cpp rev 1.02 112296 mjn pixelDyne"
#include <windows.h>
#include <windowsx.h>  //GetWindowStyleEx()
#include <stdio.h>

#include "gcmodes.h"

///////////////////////////////////////
//
// DirectDraw globals
//
///////////////////////////////////////
LPDIRECTDRAW        lpDD;           // pointer to DirectDraw object
LPDIRECTDRAWSURFACE lpDDSPrimary;   // primary surface object
LPDIRECTDRAWSURFACE lpDDSBack;      // back surface object
LPDIRECTDRAWSURFACE lpDDSOffScreen; // offscreen surface
LPDIRECTDRAWCLIPPER lpDDClipper;    // DD Clipper object
DDSURFACEDESC       gcddsd;         // game console DD surface desc

typedef struct _GCDESCtag {
  int gcWidth;    // game console width in pixels
  int gcHeight;   // game console height in pixels
  int gcbpp;      // game console screen depth in bits-per-pixel
} gcDesc;

gcDesc gcInfo;
BOOL bDDEmulate = FALSE;            // emulate DirectDraw?
BOOL bUseModeX;                     // use Mode X?
BOOL bIsFullScreen = FALSE;         // full screen EXCLUSIVE mode?
BOOL bHasBankMem;                   // bank switched video?
BOOL bUseSysMem = FALSE;            // surface is in system memory?
BOOL bOffRefBack;                   // offscreen is back buffer?
int  gcHeight;                      // console height (pixels)
int  gcWidth;                       // console width (pixels)
int  gcbpp;                         // console pixel bit depth 
RECT gcRect;                        // bounds of screen

// simple error handlers write to file
void createMsgOut(char *filename);
void appendMsgOut(char *filename, char *msg);

///////////////////////////////////////
//
// simpe message and error handlers
//
// void gcMsg( char *msg )
// BOOL gcErrMsg( char *msg, BOOL retFlag )
//
///////////////////////////////////////

///////////////////////////////////////
// text message handler
///////////////////////////////////////
void gcMsg( char *msg )
{
  MessageBox( GetActiveWindow(), msg, "gcMessage", MB_OK);
}

///////////////////////////////////////
// text message handler
///////////////////////////////////////
BOOL gcErrMsg( char *msg, BOOL retFlag )
{
  gcMsg( msg );
  return retFlag;
}

///////////////////////////////////////
// set the game console descriptor
///////////////////////////////////////
void gcSetConsole( int width, int height, int bpp, BOOL allowX )
{
  gcInfo.gcWidth = width;     // set game console width in pixels
  gcInfo.gcHeight = height;   // set game console height in pixels
  gcInfo.gcbpp = bpp;         // set game console depth in bits-per-pixel
  bUseModeX = allowX;     // full screen game console allow mode X?
}
///////////////////////////////////////
// Step 1.
// code used to initialize a 
// DirectDraw object
//
// BOOL gcInit( HWND hwnd, 
//              int width, int height, 
//              int bpp, BOOL bmodeX )
// BOOL gcSetCooperativeLevel(HWND hwnd)
// BOOL gcSetVideoMode( HWND hwnd )
// void gcFreeDD( void )
//
///////////////////////////////////////

///////////////////////////////////////
// initialize DirectDraw object
///////////////////////////////////////
BOOL gcInit( HWND hwnd)
{
  HRESULT ddrval;

  SetRect(&gcRect,0,0,gcInfo.gcWidth, gcInfo.gcHeight); // set bounds

  // create the DirectDraw object
  if (lpDD == NULL)
  {
    if (bDDEmulate)
      ddrval = DirectDrawCreate((LPGUID)DDCREATE_EMULATIONONLY, &lpDD, NULL);
	else
      ddrval = DirectDrawCreate(NULL, &lpDD, NULL);
  } 

  // DirectDraw instantian successful?
  if (ddrval != DD_OK)
    return gcErrMsg("DirectDrawCreate Failed", FALSE);
  
  // set the cooperative level with Windows
  if (gcSetCooperativeLevel(hwnd) == FALSE)
  {
    gcFreeDD();	 // free the DirectDraw object
    return FALSE;
  }

  // set the console video mode
  if (gcSetVideoMode(hwnd) == FALSE)
  {
	gcFreeDD();	 // free the DirectDraw object
    return FALSE;
  }
  if (!gcInitSurfaces(hwnd))
    return FALSE;
  return TRUE;
}

///////////////////////////////////////
// set cooperative level for GC
///////////////////////////////////////
BOOL gcSetCooperativeLevel(HWND hwnd)
{
  HRESULT ddrval;
  DWORD   dwFlags = 0;

  // set EXCLUSIVE mode for FULLSCREEN
  if (bIsFullScreen)
  {
    if (bUseModeX)
      dwFlags = DDSCL_ALLOWMODEX | DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN;
    else
      dwFlags = DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN;
  
    ddrval = lpDD->SetCooperativeLevel(hwnd,dwFlags);
    if (ddrval != DD_OK)
      return gcErrMsg("SetCooperativeLevel Failed", FALSE);
    return TRUE;
  } // if bIsFullScreen == TRUE
  else // set COOPERATIVE mode for WINDOW clipper object
  {
    dwFlags = DDSCL_NORMAL;
    ddrval = lpDD->SetCooperativeLevel(hwnd,dwFlags);
    if (ddrval != DD_OK)
      return gcErrMsg("SetCooperativeLevel Failed", FALSE);
    return TRUE;
  } // bIsFullScreen == FALSE
  return FALSE;
}

///////////////////////////////////////
// set the console video mode
///////////////////////////////////////
BOOL gcSetVideoMode( HWND hwnd )
{
  HRESULT ddrval;
  if (bIsFullScreen==TRUE)
  {
    //ddrval = lpDD->SetDisplayMode(gcInfo.gcWidth, gcInfo.gcHeight, gcInfo.gcbpp);
	ddrval = lpDD->SetDisplayMode(gcInfo.gcWidth,gcInfo.gcHeight,gcInfo.gcbpp);
    if (ddrval != DD_OK)
      return gcErrMsg("SetDisplayMode Failed", FALSE);
    return TRUE;
  }
  else  // modify WINDOW style from POPUP
  {
    if (gcSetWindowMode( hwnd ) == FALSE)
      return gcErrMsg("SetWindowMode Failed",FALSE);
    return TRUE;
  }
  return FALSE; 
}

///////////////////////////////////////
// free the DirectDraw object
///////////////////////////////////////
void gcFreeDD( void )
{
  // release the DirectDraw object
  if (lpDD != NULL)
  {
    lpDD->Release();
    lpDD = NULL;
  } // release object
}

///////////////////////////////////////
// Step 2.
// code used to initialize the
// DirectDrawSurface objects
// using triple buffering
//
// BOOL gcCreateSurfaces( void )
// void gcSetDesc( DWORD dwBufCount, 
//                 DWORD dwFlags, 
//                 DWORD dwCaps )
// BOOL gcGetBackBuffer( void )
// void gcFreeSurfaces( void )
// 
///////////////////////////////////////

BOOL gcInitSurfaces(HWND hwnd)
{
  // initialize the DirectDrawSurfaces
  if (bIsFullScreen)
  {
    if (!gcCreateSurfaces())
      return FALSE;
    if (!gcGetBackBuffer())
      return FALSE;
  }
  else
  {
    if (!gcCreateClipper(hwnd))
      return FALSE;
  }
  if (!gcCreateOffScreen())
    return FALSE;
  return TRUE;
}

///////////////////////////////////////
// create the DirectDrawSurface(s)
///////////////////////////////////////
BOOL gcCreateSurfaces( void )
{
  DWORD         dwFlags = 0;
  DWORD         dwCaps = 0;
  HRESULT       ddrval;

  createMsgOut("flip.out"); // create output file

  lpDDSPrimary = NULL;
  lpDDSBack = NULL;

  // try a for a triple buffer in video memory
  appendMsgOut("flip.out","Creating a triple buffer\n"); // debug:
  dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
  dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP
	  | DDSCAPS_COMPLEX | DDSCAPS_VIDEOMEMORY;
  gcSetDesc( TRIPLE_BUFFER, dwFlags, dwCaps );
  ddrval = lpDD->CreateSurface( &gcddsd, &lpDDSPrimary, NULL );
  // did we get a triple buffer in video memory?
  if (ddrval != DD_OK)
  {
    gcErrMsg( "TRIPLE BUFFER FAILED", FALSE );
	appendMsgOut("flip.out","\t\ttriple buffer FAILED\n"); // debug:
	// go for a double buffer
    // try a for a double buffer in video memory
    appendMsgOut("flip.out","Creating a double buffer(VIDOEMEM)\n"); // debug:
	gcSetDesc( DOUBLE_BUFFER, dwFlags, dwCaps );
    ddrval = lpDD->CreateSurface( &gcddsd, &lpDDSPrimary, NULL );
    if (ddrval != DD_OK)
    {
      gcErrMsg( "DOUBLE BUFFER FAILED", FALSE );
	  appendMsgOut("flip.out","\t\tdouble buffer FAILED\n"); // debug:
      // try a for a double buffer using system memory
      appendMsgOut("flip.out","Creating double buffer (SYSMEM)\n"); // debug:
	  dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP
	    | DDSCAPS_COMPLEX;
      gcSetDesc( DOUBLE_BUFFER, dwFlags, dwCaps );
      ddrval = lpDD->CreateSurface( &gcddsd, &lpDDSPrimary, NULL );
	  if (ddrval == DD_OK)
	    bUseSysMem = TRUE;
    }
  }

  DDSCAPS ddscaps;
  ddrval = lpDDSPrimary->GetCaps(&ddscaps);
  if (ddrval != DD_OK)
	appendMsgOut("flip.out","\t\tGetCaps FAILED\n");

  if ( ddscaps.dwCaps & DDSCAPS_MODEX)
  {
    appendMsgOut("flip.out","  Using ModeX\n");
  }
  else 
    appendMsgOut("flip.out","  Not using ModeX\n");

  return TRUE;
}

///////////////////////////////////////
// set the gc DDSURFACEDESC fields
///////////////////////////////////////
void gcSetDesc( DWORD dwBufCount, DWORD dwFlags, DWORD dwCaps )
{
  memset( &gcddsd, 0, sizeof(gcddsd) );

  gcddsd.dwSize = sizeof( gcddsd );
  gcddsd.dwFlags = dwFlags;
  gcddsd.dwBackBufferCount = dwBufCount;
  gcddsd.ddsCaps.dwCaps = dwCaps;
}

///////////////////////////////////////
// attache the back buffer
///////////////////////////////////////
BOOL gcGetBackBuffer( void )
{
  HRESULT ddrval;
  DDSCAPS ddscaps;

  ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
  ddrval = lpDDSPrimary->GetAttachedSurface( &ddscaps, &lpDDSBack ); 
  if (ddrval != DD_OK)
    return FALSE;
  return TRUE;
}

///////////////////////////////////////
// gc release surface(s)
///////////////////////////////////////
void gcFreeSurfaces( void )
{
  // free offscreen if it doesn't reference lpDDSBack
  if(lpDDSOffScreen != lpDDSBack && lpDDSOffScreen != NULL)
  {
    lpDDSOffScreen->Release();
    lpDDSOffScreen = NULL;
  }
  // free the back buffer
  if (lpDDSBack != NULL)
  {
    lpDDSBack->Release();
    lpDDSBack = NULL;
  }
  // free the clipper object
  if (lpDDClipper != NULL)
  {
    lpDDClipper->Release();
    lpDDClipper = NULL;
  }
  // free the primary surface
  if (lpDDSPrimary != NULL)
  {
    lpDDSPrimary->Release();
    lpDDSPrimary = NULL;
  }
}

///////////////////////////////////////
// DirectDraw Surface Blit Utility
///////////////////////////////////////
void gcEraseSurface(LPDIRECTDRAWSURFACE lpDDSurfPtr)
{
  DDBLTFX ddbltfx;
  ddbltfx.dwSize = sizeof(ddbltfx);
  ddbltfx.dwFillColor = 0;
  HRESULT ddrval;

  while (1)
  {
    ddrval = lpDDSurfPtr->Blt(NULL,NULL,NULL,DDBLT_COLORFILL,&ddbltfx);
    if (ddrval == DD_OK)
      break;
    if (ddrval ==DDERR_SURFACELOST)
    { 
      ddrval = gcRestoreSurfaces();
      if (ddrval != DD_OK)
        return;
    }

    if (ddrval != DDERR_WASSTILLDRAWING )
      return;
  }
}


///////////////////////////////////////
// Shutdown game console
///////////////////////////////////////
void gcFree( void )
{
  // free the clipper and surfaces
  gcFreeSurfaces();

  // free the GC DirectDraw object
  gcFreeDD();
}

///////////////////////////////////////
// Step 3.
// code used to initialize the
// DirectDrawSurface object
// offscreen surface
//
//  BOOL gcCreateOffScreen( void )
// 	IDirectDrawSurface*  
//       gcGetNewSurface( int width, 
//                        int height, 
//                        BOOL bSysMem )
//  BOOL gcCheckIsBankMem( void )
//
///////////////////////////////////////

///////////////////////////////////////
// Create lpDDSOffscreen 
///////////////////////////////////////
BOOL gcCreateOffScreen( void )
{
  DDSCAPS       ddscaps;
  HRESULT       ddrval;

  // debug: write values to disk
  appendMsgOut("flip.out","_gcCreateOffScreen\n");

  ddrval = lpDDSBack->GetCaps(&ddscaps);
  if (ddrval != DD_OK)
  {
	appendMsgOut("flip.out","\t\tGetCaps FAILED\n");
    return FALSE;
  }

  bHasBankMem = gcCheckIsBankMem();
  // if device is bankswitched and lpDDSBack in video memory
  // create a system offscreen buffer
  if (bHasBankMem && (ddscaps.dwCaps & DDSCAPS_VIDEOMEMORY))
  {
    bOffRefBack = FALSE;
    if (lpDDSOffScreen == NULL)
    {
      appendMsgOut("flip.out","  FAILED lpDDSOffScreen Alloc (bankswitched)\n");
      return FALSE;   // lpDDSOffScreen allocation failed
    }
    else
    {
      appendMsgOut("flip.out","  lpDDSOffScreen Alloc OKAY (bankswitched)\n");
      return TRUE;    // allocation was succesful
    }
  }

  // if device NOT bankswitched, render directly to lpDDSBack
  bOffRefBack = TRUE;
  lpDDSOffScreen = lpDDSBack;

  appendMsgOut("flip.out","  lpDDSOffScreen => lpDDSBack\n");
  return TRUE;
}

///////////////////////////////////////
// Create a NEW unattached surface
///////////////////////////////////////
IDirectDrawSurface*  gcGetNewSurface( int width, int height, BOOL bSysMem )
{
  IDirectDrawSurface* lpDDSurface;
  DDSURFACEDESC       ddsd;
  HRESULT             ddrval;

  memset( &ddsd, 0, sizeof(ddsd) );
  ddsd.dwSize = sizeof( ddsd );
  ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
  if (bSysMem)
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
  else
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
  ddsd.dwHeight = height;
  ddsd.dwWidth = width;
  ddrval = lpDD->CreateSurface(&ddsd, &lpDDSurface, NULL);
  if (ddrval != DD_OK)
    return NULL;

  return (IDirectDrawSurface*) lpDDSurface; 
}

///////////////////////////////////////
// Check if vram is bank switched
///////////////////////////////////////
BOOL gcCheckIsBankMem( void )
{
  DDCAPS ddcaps;
  HRESULT ddrval;

  // debug: write values to disk
  appendMsgOut("flip.out","_gcCheckIsBankMem\n");

  ddrval = lpDD->GetCaps(&ddcaps, NULL);
  if ( ddcaps.dwCaps & DDCAPS_BANKSWITCHED)
  {
    appendMsgOut("flip.out","\t\tVideo Device is bankswitched\n");
    return TRUE;
  }

  appendMsgOut("flip.out","\t\tVideo Device is not bankswitched\n");
  return FALSE;
}

///////////////////////////////////////
// 
// code used to update
// the primary DirectDrawSurface.
// * if back buffer is in vram
//      -> surface flip
// * if back buffer is in sys mem
//      -> blt < back to primary >
//
//  void gcUpdateSurface( void ) 
//	HRESULT gcRestoreSurfaces( void )
//
///////////////////////////////////////

///////////////////////////////////////
// update the primary surface
///////////////////////////////////////
void gcUpdateSurface( void )
{
  HRESULT ddrval;
  if (bIsFullScreen)
  {
    ddrval = lpDDSPrimary->Flip( NULL, DDFLIP_WAIT );
    if (ddrval == DD_OK)
      return;
    if (ddrval == DDERR_SURFACELOST)
    {
      ddrval = gcRestoreSurfaces();
      if (ddrval != DD_OK)
        return;
    }
    return;
  } 
  if (!bIsFullScreen) // Handle updating a WINDOW
  {
    ddrval = lpDDSPrimary->Blt(&gcRect, lpDDSBack, NULL, DDBLT_WAIT, NULL);
    if (ddrval != DD_OK)
      return;
  }
  return;
}

///////////////////////////////////////
// Restore DirectDrawSurfaces
///////////////////////////////////////
HRESULT gcRestoreSurfaces( void )
{
  HRESULT ddrval = lpDDSPrimary->Restore();
  if (ddrval == DD_OK)
    ddrval = lpDDSBack->Restore();
  if (ddrval == DD_OK)
    ddrval = lpDDSOffScreen->Restore();
  return ddrval;
}

///////////////////////////////////////
// 
// code used to create the
// WINDOW game console clipper
//
///////////////////////////////////////

///////////////////////////////////////
//  Setup a window to use for the GC
///////////////////////////////////////
BOOL gcSetWindowMode( HWND hwnd )
{
  RECT  rc;
  DWORD dwStyle;

  // get the current window style
  //  a FULLSCREEN gc uses WS_POPUP
  //  a WINDOWed gc uses normal window styles
  dwStyle = GetWindowStyle(hwnd);
  dwStyle &= ~WS_POPUP;  // remove WS_POPUP style
  dwStyle |= WS_OVERLAPPED | WS_CAPTION | WS_MINIMIZEBOX;
  SetWindowLong(hwnd, GWL_STYLE, dwStyle);
  SetRect(&rc, 0, 0, gcInfo.gcWidth, gcInfo.gcHeight);
  AdjustWindowRectEx(&rc,
                     GetWindowStyle(hwnd),
                     GetMenu(hwnd) != NULL,
                     GetWindowExStyle(hwnd));
  SetWindowPos(hwnd, NULL, 0, 0, rc.right - rc.left, rc.bottom-rc.top,
             SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);
  SetWindowPos(hwnd, HWND_NOTOPMOST,0,0,0,0,
             SWP_NOSIZE | SWP_NOMOVE | SWP_NOACTIVATE);

  RECT rcWork;
  SystemParametersInfo(SPI_GETWORKAREA, 0, &rcWork, 0);
  GetWindowRect(hwnd, &rc);
  if (rc.left < rcWork.left)
    rc.left = rcWork.left;
  if (rc.top < rcWork.top)
    rc.top = rcWork.top;
  SetWindowPos(hwnd, NULL, rc.left, rc.top, 0, 0,
      SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE );

  GetWindowRect(hwnd, &gcRect); // adjust and align gcRect for window
  return TRUE;
}

///////////////////////////////////////
//  Setup a window to use for the GC
///////////////////////////////////////
BOOL gcCreateClipper( HWND hwnd )
{
  HRESULT ddrval;

  // debug: write values to disk
  createMsgOut("flip.out");
  gcSetDesc( 0, DDSD_CAPS, DDSCAPS_PRIMARYSURFACE );

  ddrval = lpDD->CreateSurface(&gcddsd, &lpDDSPrimary, NULL);
  if (ddrval != DD_OK)
  {
    appendMsgOut("flip.out","  Create GC_WINDOW lpDDSPrimary Failed\n");
    return FALSE;
  }

  appendMsgOut("flip.out","Create the GC_WINDOW lpDDSBack\n");
  lpDDSBack = gcGetNewSurface(gcInfo.gcWidth, gcInfo.gcHeight, FALSE);
  if (lpDDSBack == NULL)
  {
    appendMsgOut("flip.out","  Create GC_WINDOW lpDDSBack Failed\n");
    return FALSE;
  }

  bUseSysMem = TRUE; // back buffer in system memory
  appendMsgOut("flip.out","Create the GC_WINDOW lpDDClipper\n");
  ddrval = lpDD->CreateClipper(0, &lpDDClipper, NULL);
  if (ddrval != DD_OK)
  {
    appendMsgOut("flip.out","  Create GC_WINDOW Clipper object Failed\n");
    return FALSE;
  }

  ddrval = lpDDClipper->SetHWnd(0,hwnd);
  if (ddrval != DD_OK)
  {
    appendMsgOut("flip.out","  Create GC_WINDOW ::SetHWnd Failed\n");
    return FALSE;
  }

  ddrval =  lpDDSPrimary->SetClipper(lpDDClipper);
  if (ddrval != DD_OK)
  {
    appendMsgOut("flip.out","  Create GC_WINDOW ::SetClipper object Failed\n");
    return FALSE;
  }
  return TRUE;
}

///////////////////////////////////////
//  Toggle between Window and Video
///////////////////////////////////////
BOOL gcToggleWindow(HWND hwnd)
{
  BOOL bResult;

  if(lpDD != NULL )    // if a DirectDraw object exists
    gcFree();  //   destroy it

  if (bIsFullScreen == TRUE) // toggle from fullscreen to window mode
  {                          // were we using fullscreen video mode?
    bIsFullScreen = FALSE;   // set console flag for window mode
    bResult = gcInit(hwnd);
	if (bResult == FALSE)    // set the window mode
	  return FALSE;          // did we succeed, no? fatal error
	else                     // mode toggle succeeded 
	  return TRUE;           // everythings good, return
  }             
  else                       // toggle from a window to fullscreen
  {                          // we were executing in a window
    bIsFullScreen = TRUE;    // set flag console to full screen mode
    SetRect(&gcRect,0,0,gcInfo.gcWidth, gcInfo.gcHeight); // set bounds
  }
  return (gcInit(hwnd));
}

//
// debug
//
void createMsgOut(char *filename)
{
  FILE *fdebug = fopen(filename,"w+");
  fclose(fdebug);
}

void appendMsgOut(char *filename, char *msg)
{
  FILE *fdebug;
  fdebug = fopen(filename,"a+");
  fprintf(fdebug,msg);
  fclose(fdebug);
}

