/* 
MEMPROBE.C -- Win32 Console application to test memory protection

A Win32 program to determine the actual address space in use by Win32
programs in Windows 95. Very different results from those shown by
the VirtualQuery API! MEMPROBE walks the entire four-gigabyte
potential address space, using the structured exception handling
(SEH) _try and _except statements to determine which areas are
readable and writeable, read-only (write-protected), or protected
(neither readable nor writeable). Alternatively, if compiled with the
USE_SEH switch, MEMPROBE will use the IsBadWritePtr and IsBadReadPtr
APIs. Right now, MEMPROBE.EXE is a Console app, but can be compiled
with the NOT_CONSOLE switch if you want to try out MEMPROBE without
needing an extra virtual machine (VM) for the Console output window.

Andrew Schulman
andrew@ora.com
http://www.ora.com/windows/
ftp://ftp.ora.com/pub/examples/windows/win95.update/schulman.html
August 1995

The program generates a MEMPROBE.LOG file which looks like this:
00010000h-00124000h Read/Write (1130496 bytes)
0019F000h-0026E000h Read/Write (847872 bytes) 
00400000h-00409000h Read 
00409000h-0040B000h Read/Write (8192 bytes) 
0040B000h-0040C000h Read 
0040C000h-0040F000h Read/Write (12288 bytes) 
0040F000h-00410000h Read 
00410000h-00411000h Read/Write (4096 bytes) 
... etc., etc. ...
C1F4F000h-C1F50000h Read/Write (4096 bytes) 
C234E000h-C234F000h Read/Write (4096 bytes) 
C2400000h-C2524000h Read/Write (1196032 bytes) 
C259F000h-C25A0000h Read/Write (4096 bytes) 
552 seconds
24346624 pokeable bytes out of 100000000h 
0.57% pokeable
*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "excpt.h"

#define USE_SEH

typedef enum { 
    NONE = 0, BAD_READ=1, BAD_WRITE=2, BAD_RW=3, GOOD_RW=4,
    } rgn;

FILE *logfile;

#ifdef NOT_CONSOLE
#define PUTS(s)         MessageBox(0, (s), "MEMPROBE", MB_OK)
#else
#define PUTS(s)         puts(s)
#endif

void fail(const char *s) { PUTS(s); exit(1); }

void display_range(rgn state, int *start, unsigned size)
{
    static char *name[5] = { 
        "Internal error!", "Write", "Read", "Protected", "Read/Write"
        } ;
    unsigned ul = (unsigned) start;
    
    if (! state) return;
    if (state == BAD_RW) return;    // don't show "Protected"
        
    fprintf(logfile, "%08lXh-%08lXh\t%s", 
        ul, ul+(size*sizeof(int)), name[state]);
    if (state == GOOD_RW) fprintf(logfile," (%lu bytes)", size*sizeof(int));
    fprintf(logfile,"\n");
    fflush(logfile);
#ifndef NOT_CONSOLE 
    fprintf(stderr, "%08lXh-%08lXh\t%s", 
        ul, ul+(size*sizeof(int)), name[state]);
    if (state == GOOD_RW) fprintf(stderr," (%lu bytes)", size*sizeof(int));
    fprintf(stderr,"\n");
#endif
}

#ifdef NOT_CONSOLE
#define argc __argc
#define argv __argv
extern int __argc;
extern char **__argv;

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance,
    LPSTR lpszCmdLine, int nCmdShow)
#else
main(int argc, char *argv[])
#endif
{
    unsigned size, pokeable_bytes = 0, bytes;
    rgn state = NONE, newst = NONE;
    int r, w, x, *p, *start, *end, *begin;
    char *filename;
    time_t t1, t2;
    
    if (argc < 2) p = (int *) 0;
    else sscanf(argv[1], "%x", (unsigned *) &p);  
    begin = p;
    if (argc < 3) end = (int *) 0;
    else sscanf(argv[2], "%x", (unsigned *) &end);
    if (end < begin)    // interpret as length
        end = (int *) (((BYTE *) end) += (DWORD) begin);
    filename = (argc < 4) ? "memprobe.log" : argv[3];
    logfile = fopen(filename, "w");
    bytes = 0;
    
    if (! logfile) fail("Can't create log file");
    
    time(&t1);
    
    do {
#ifdef USE_SEH      
        // use Structured Exception Handling (SEH)
        _try { r = 0; x = *p; }                             // try peek
        _except (EXCEPTION_EXECUTE_HANDLER) { r = 1; }      // peek failed
            
        // because we only write to write if we could read,
        // the BAD_READ state never happens

        if (r == 0)     // if could successfully peek
        {
            _try { w = 0; x = *p; *p = x; }                 // try poke
            _except (EXCEPTION_EXECUTE_HANDLER) { w = 1; } // poke failed
            if (*p != x) w = 1;                    // didn't really write?
        }
        else w = 1;     // if not readable, assume not writeable
#else
        // use the Win32 IsBadXXXXPtr API functions
        if ((r = IsBadReadPtr(p, sizeof(int))) == 0)    // if readable
            w = IsBadWritePtr(p, sizeof(int));          // see if writeable
        else w = 1;
#endif      
        
        newst = (r && w) ? BAD_RW : r ? BAD_READ : w ? BAD_WRITE : GOOD_RW;
        
        if (newst == state)
            size += 1024;
        else
        {
            display_range(state, start, size);
            
            if (state == GOOD_RW) pokeable_bytes += (size * sizeof(int));
            
            start = p;
            size = 1024;
            state = newst;
        }

        p += 1024;      // do by pages
        bytes += 4096;
    } while (p != end);
    
    time(&t2);
    
    // get last one
    display_range(state, start, size);
    
    fprintf(logfile, "%lu seconds\n", t2-t1);
    if (bytes)
    {
        fprintf(logfile, "%lu pokeable bytes out of %lu\n", 
            pokeable_bytes, bytes);
        fprintf(logfile, "%0.2f%% pokeable\n",
            (double) (pokeable_bytes * 100.0) / (double) bytes);
    }
    else
    {
        fprintf(logfile, "%lu pokeable bytes out of 100000000h\n", 
            pokeable_bytes, bytes);
        fprintf(logfile, "%0.2f%% pokeable\n",
            (double) (pokeable_bytes * 100.0) / ((double) 0xFFFFFFFF + 1));
    }
    
#ifdef NOT_CONSOLE
    MessageBox(0, "Done! See MEMPROBE.LOG", "MEMPROBE", MB_OK);
#else
    fprintf(stderr, "\nDone! See MEMPROBE.LOG\n");
#endif
    
    fclose(logfile);

    return 0;
}
