///////////////////////////////////////////////////////////////////////////////
//Brad Johnson Console Telnet a Win32 ANSI telnet client.
//Copyright (C) 1997  Brad Johnson
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//jbj@chrysalis.org
//brad.johnson@chrysalis.org
//
//Brad Johnson
//P.O. Box 933
//Crowley, TX 76036
//
///////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
// Module:		tnparser.cpp
//
// Contents:	telnet parser screen functions
//
// Product:		telnet
//
// Revisions: 05. Sep.1997 roryt@hol.gr (I.Ioannou)
//            11.May,1997   roryt@hol.gr
//            06.April,1997 roryt@hol.gr
//            30.Mrz.1997  Titus_Boxberg@public.uni-hamburg.de
//		        5.Dec.1996 jbj@chrysalis.org
//            Version 2.0
//            02.Apr.1995	igor.milavec@uni-lj.si
//					  Original code
//
///////////////////////////////////////////////////////////////////////////////

#include "tnparser.h"

#define CON_TOP 0
#define CON_BOTTOM (ConsoleInfo.dwSize.Y - 1)
#define CON_LEFT 0
#define CON_RIGHT (ConsoleInfo.dwSize.X - 1)
#define CON_HEIGHT (CON_BOTTOM - CON_TOP)
#define CON_WIDTH (CON_RIGHT - CON_LEFT)
#define CON_LINES (CON_HEIGHT + 1)
#define CON_COLS (CON_WIDTH + 1)

#define CON_CUR_X (DWORD)(ConsoleInfo.dwCursorPosition.X - CON_LEFT)
#define CON_CUR_Y (DWORD)(ConsoleInfo.dwCursorPosition.Y - CON_TOP)


TTelnetParser::TTelnetParser(HANDLE hConsole, SOCKET Socket, char * szDumpFileName,
   char * pcSD, long iSS, long * iSB):
	hConsole(hConsole), Socket(Socket), TANSIParser(szDumpFileName,pcSD,iSS,iSB)

{
  CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
  GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo);
  wAttributes = (unsigned char) 7;      //LO WHITE;
  iNumOfLines = CON_LINES;
  iNumOfCols = CON_COLS;
  SetConsoleTextAttribute(hConsole,wAttributes);
  bLineWrapEnabled = 1;
//  printf("\nCON_BOTTOM=%i CON_RIGHT=%i\n", CON_BOTTOM, CON_RIGHT);
  InReverseMode = 0;
}

TTelnetParser::~TTelnetParser()
{
  wAttributes=(unsigned char) 7;      //LO WHITE;
  ConSetCursorPosition( 0, iNumOfLines-1);
  SetConsoleTextAttribute( hConsole, wAttributes);
  ConWriteCtrlString("\x0a");
}

void TTelnetParser::ConSetAttrib(unsigned char wAttr)
{
  wAttributes = wAttr;
}

unsigned char TTelnetParser::ConGetAttrib()
{
  return wAttributes;
}

void TTelnetParser::ConHighVideo()
{
  wAttributes = wAttributes | (unsigned char) 8;
}

void TTelnetParser::ConLowVideo()
{
  wAttributes = wAttributes & (unsigned char) (0xff-8);
}

void TTelnetParser::ConNormal()
{
// I.Ioannou 11 May 1997
// FIX ME. The color 7 is correct on some systems (for example Linux)
// but on others (for example SCO) we must preserve the colors :
// A setup parameter maybe ??

#if !defined(PRESERVE_COLORS)
  wAttributes= (unsigned char) 7;
  InReverseMode = 0;
#else
  ConBlinkOff();
  ConReverseOff();
  ConLowVideo();
#endif // NORMAL_PRESERVE_ATTR
}

void TTelnetParser::ConForground(unsigned char wAttrib)
{
  wAttributes= (wAttributes & (unsigned char) 0xf8)| wAttrib;
}

void TTelnetParser::ConBackground(unsigned char wAttrib)
{
  wAttributes= (unsigned char)(wAttributes & (unsigned char) 0x8f)|(unsigned char) (wAttrib << 4);
}

void TTelnetParser::ConBlink()
{
  wAttributes= wAttributes | (unsigned char)0x80;
}

// Added by I.Ioannou 06 April, 1997
void TTelnetParser::ConBlinkOff()
{
  wAttributes= wAttributes & (unsigned char)0x7f;
}

// Added by I.Ioannou 11.May,1997
void TTelnetParser::ConReverseOn()
{
  if ( ! InReverseMode )
  {
    InReverseMode = 1;

    // atl  : forground attributes without the intensity
    // ath  : backgound attributes without the blink
    // bl   : the blink state
    // ints : the intensity
    unsigned char atl   = wAttributes & (unsigned char) 0x07;
    unsigned char ath   = wAttributes & (unsigned char) 0x70;
    unsigned char bl    = wAttributes & (unsigned char) 0x80;
    unsigned char ints  = wAttributes & (unsigned char) 0x08;
    wAttributes = bl | (atl << 4) | ints | (ath >> 4);
  }
}

// Added by I.Ioannou 11.May,1997
void TTelnetParser::ConReverseOff()
{
  if ( InReverseMode )
  {
    InReverseMode = 0;

    // atl  : forground attributes without the intensity
    // ath  : backgound attributes without the blink
    // bl   : the blink state
    // ints : the intensity
    unsigned char atl   = wAttributes & (unsigned char) 0x07;
    unsigned char ath   = wAttributes & (unsigned char) 0x70;
    unsigned char bl    = wAttributes & (unsigned char) 0x80;
    unsigned char ints  = wAttributes & (unsigned char) 0x08;
    wAttributes = bl | (atl << 4) | ints | (ath >> 4);
  }
}

int TTelnetParser::ConWriteString(char* pszString, int cbString)
{
	DWORD Result;
  CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
  GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo);
  int iCurCols = CON_CUR_X;
  SetConsoleTextAttribute(hConsole,wAttributes);

  //check to see if the line is longer than the display
  if (!bLineWrapEnabled && (iCurCols + cbString) >= iNumOfCols){
    // Take care of the last line last colum exception...
    // The display scrolls up if you use the normal char out
    //   function even if you only write to the last place
    //   on the line. :-(
    if ( CON_CUR_Y >= CON_BOTTOM ){
      int iFakeResult = cbString;
      cbString = CON_COLS - CON_CUR_X - 1;
      WriteConsole(hConsole, pszString, cbString, &Result, 0);

      COORD dwBufferCoord;
      dwBufferCoord.X = CON_LEFT;
      dwBufferCoord.Y = CON_TOP;

      CHAR_INFO ciBuffer;
      ciBuffer.Char.AsciiChar = *(pszString+cbString);
      ciBuffer.Attributes = wAttributes;
      SMALL_RECT srWriteRegion;
      srWriteRegion.Top =   (SHORT) CON_BOTTOM;
      srWriteRegion.Bottom =(SHORT) CON_BOTTOM;
      srWriteRegion.Left =  (SHORT) CON_RIGHT;
      srWriteRegion.Right = (SHORT) CON_RIGHT;

      WriteConsoleOutput( hConsole, &ciBuffer, ConsoleInfo.dwSize,
        dwBufferCoord, &srWriteRegion );
      return iFakeResult; // Skip the chars that did not fit
    }
    // just write the line up to the end
    else {
      int iFakeResult = cbString;
      cbString = iNumOfCols - iCurCols ;
      WriteConsole(hConsole, pszString, cbString, &Result, 0);
      return iFakeResult; // Skip the chars that did not fit
    }
  }
  else
  // normal line will wrap normaly or not to the end of buffer
    WriteConsole(hConsole, pszString, cbString, &Result, 0);
	return Result;
}

int TTelnetParser::ConWriteCtrlString(char* pszString)
{
  // The console does not handel the CR/LF chars as we might expect
  // when using color. The attributes are not set correctly, so we
  // must interpret them manualy to preserve the colors on the screen.
  CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
  GetConsoleScreenBufferInfo(hConsole,&ConsoleInfo);
  DWORD Result;
  switch ( *pszString){
    case '\x09': // horizontal tab
//      ConSetCursorPosition( (((ConGetCursorX()/8)+1)*8) ,ConGetCursorY());
      ConSetCursorPosition( (((CON_CUR_X/8)+1)*8), CON_CUR_Y);
    break;

    case '\x0a': // line feed
      if (CON_CUR_Y >= iNumOfLines-1) { // if on the last line scroll up
        CHAR_INFO ciChar;
        COORD dwDestOrg = {0,-1};             // shift the screen up one line
        SMALL_RECT srBuf;
        srBuf.Top =    (SHORT) CON_TOP;
        srBuf.Left =   (SHORT) CON_LEFT;
        srBuf.Bottom = (SHORT) CON_BOTTOM;
        srBuf.Right =  (SHORT) CON_RIGHT;

        ciChar.Char.AsciiChar= ' ';           // fill with spaces
        ciChar.Attributes= wAttributes;       // fill with current attrib

        ScrollConsoleScreenBuffer(hConsole,
          &srBuf, // scroll rect
          &srBuf, //clip rect
          dwDestOrg, &ciChar);
      }
      else       // else move cursor down to the next line
        ConSetCursorPosition(CON_CUR_X,CON_CUR_Y+1);
     	Result = 1;
      break;
    case '\x0d': // carrage return
      ConSetCursorPosition(0,CON_CUR_Y); // move to the beginning of line
     	Result = 1;
      break;
    default :    // else just write it like normal
      break;
  }
	return Result;
}

void TTelnetParser::ConScrollDown( int iStartRow , int iEndRow, int bUp ){
  CHAR_INFO ciChar;
  COORD dwDestOrg = {0,0};
  SMALL_RECT srScrollWindow;
  CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;

  // Correction from I.Ioannou 11 May 1997
  // check the scroll region
  if ( iStartRow < iScrollStart)
      iStartRow = iScrollStart;
  if ( iEndRow > iScrollEnd)
      iEndRow = iScrollEnd;

  if (bUp)
    dwDestOrg.Y =  1;             // shift the screen up one line
  else
    dwDestOrg.Y =  -1;             // shift the screen Down one line

  GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo);

  // Correction from I.Ioannou 11 May 1997
  // this will make Top the CON_TOP
  if ( iStartRow == -1)
     iStartRow = 0;
  //
  // Correction from I.Ioannou 18 Aug 97
  if ( iEndRow == -1)
  {
    if ( iScrollEnd == -1 )
    {
      iEndRow = CON_HEIGHT;
    }
    else
      iEndRow = ((CON_HEIGHT <= iScrollEnd) ? CON_HEIGHT : iScrollEnd);
   }
//
  if ( iStartRow > CON_HEIGHT) iStartRow = CON_HEIGHT;
  if ( iEndRow > CON_HEIGHT) iEndRow = CON_HEIGHT;

  srScrollWindow.Left =           CON_LEFT;
  srScrollWindow.Right =  (SHORT) (CON_WIDTH);
  srScrollWindow.Top =    (SHORT) (CON_TOP + iStartRow );
  srScrollWindow.Bottom = (SHORT) (CON_TOP + iEndRow - 1);


  ciChar.Char.AsciiChar = ' ';           // fill with spaces
  ciChar.Attributes = wAttributes;       // fill with current attrib

  // correction by I.Ioannou 06 April, 1997
  // the Win32 Api manual says that dwDestOrg is the destination ,
  // not a relative position
  dwDestOrg.Y += srScrollWindow.Top;
  dwDestOrg.X += srScrollWindow.Left;

  ScrollConsoleScreenBuffer(hConsole, &srScrollWindow,
    NULL,dwDestOrg, &ciChar);
}

void TTelnetParser::ConLineWrap( int bEnabled ){
  bLineWrapEnabled = bEnabled;
}

void TTelnetParser::ConClearScreen()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = {CON_LEFT, CON_TOP};
	FillConsoleOutputCharacter(hConsole, ' ', (DWORD)(CON_COLS)*
		(DWORD)(CON_LINES), Coord, &dwWritten);
   FillConsoleOutputAttribute(hConsole, wAttributes, (DWORD)(CON_COLS)*
		(DWORD)(CON_LINES), Coord, &dwWritten);
}

void TTelnetParser::ConClearEOScreen()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = {CON_LEFT, (SHORT)(CON_CUR_Y + 1)};
	FillConsoleOutputCharacter(hConsole, ' ', (DWORD)(CON_COLS)*
		(DWORD)(CON_HEIGHT - CON_CUR_Y), Coord, &dwWritten);
   FillConsoleOutputAttribute(hConsole, wAttributes, (DWORD)(CON_COLS)*
		(DWORD)(CON_LINES - CON_CUR_Y), Coord, &dwWritten);
	ConClearEOLine();
}

void TTelnetParser::ConClearBOScreen()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = {CON_LEFT, CON_TOP};
	FillConsoleOutputCharacter(hConsole, ' ', (DWORD)(CON_COLS)*
		(DWORD)(CON_HEIGHT - CON_CUR_Y ), Coord, &dwWritten);
   FillConsoleOutputAttribute(hConsole, wAttributes, (DWORD)(CON_COLS)*
		(DWORD)(CON_HEIGHT - CON_CUR_Y ), Coord, &dwWritten);
	ConClearBOLine();
}

void TTelnetParser::ConClearLine()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = { CON_LEFT, (SHORT)(CON_TOP + CON_CUR_Y)};
	FillConsoleOutputCharacter(hConsole, ' ',	(DWORD)(CON_WIDTH),
		Coord, &dwWritten);
   FillConsoleOutputAttribute(hConsole, wAttributes, (DWORD)(CON_WIDTH),
		Coord, &dwWritten);
}

void TTelnetParser::ConClearEOLine()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
   COORD Coord = {(SHORT)(CON_LEFT + CON_CUR_X),(SHORT)(CON_TOP + CON_CUR_Y)};
   FillConsoleOutputAttribute(hConsole, wAttributes,
      (DWORD)(CON_RIGHT - CON_CUR_X) +1, Coord, &dwWritten);
   FillConsoleOutputCharacter(hConsole, ' ',	(DWORD)(CON_RIGHT - CON_CUR_X) +1,
	   Coord, &dwWritten);
}

void TTelnetParser::ConClearBOLine()
{
	DWORD dwWritten;
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = { CON_LEFT,(SHORT)(CON_TOP + CON_CUR_Y)};
	FillConsoleOutputCharacter(hConsole, ' ',(DWORD)(CON_CUR_X), Coord,
      &dwWritten);
  FillConsoleOutputAttribute(hConsole, wAttributes,(DWORD)(CON_CUR_X),
		Coord, &dwWritten);
}


//	Inserts blank lines to the cursor-y-position
//	scrolls down the rest. CURSOR MOVEMENT (to Col#1) ???
void TTelnetParser::InsertLine(int numlines)
{
  COORD		to;
  SMALL_RECT	from;
  SMALL_RECT	clip;
  CHAR_INFO		fill;
  int		acty;

// Rest of screen would be deleted
  if ( (acty = ConGetCursorY()) >= iNumOfLines - numlines ) {
    ConClearEOScreen();				// delete rest of screen
    return;
  } /* IF */

//	Else scroll down the part of the screen which is below the
//	cursor.
  from.Left = 0;
  from.Top = (SHORT)acty;
  from.Right = (SHORT)iNumOfCols;
  from.Bottom = (SHORT)iNumOfLines;

  clip = from;
  to.X = 0;
  to.Y = (SHORT)(from.Top + numlines);

  fill.Char.AsciiChar = ' ';
  fill.Attributes = 7; 		// WHICH ATTRIBUTES TO TAKE FOR BLANK LINE ??

  ScrollConsoleScreenBuffer(hConsole, &from, &clip, to, &fill);
} /* InsertLine */

//	Inserts blank characters under the cursor
void TTelnetParser::InsertCharacter(int numchar)
{
  int		actx;
  SMALL_RECT	from;
  SMALL_RECT	clip;
  COORD			to;
  CHAR_INFO		fill;

  if ( (actx = ConGetCursorX()) >= iNumOfCols - numchar ) {
    ConClearEOLine();
    return;
  } /* IF */

  from.Left = (SHORT)actx;
  from.Top = (SHORT)ConGetCursorY();
  from.Right = (SHORT)iNumOfCols;
  from.Bottom = (SHORT)from.Top;

  clip = from;
  to.X = (SHORT)(actx + numchar);
  to.Y = from.Top;

  fill.Char.AsciiChar = ' ';
  fill.Attributes = wAttributes; // WHICH ATTRIBUTES TO TAKE FOR BLANK CHAR ??

  ScrollConsoleScreenBuffer(hConsole, &from, &clip, to, &fill);
} /* InsertCharacter */

//	Deletes characters under the cursor
void TTelnetParser::DeleteCharacter(int numchar)
{
  int		actx;
  SMALL_RECT	from;
  SMALL_RECT	clip;
  COORD			to;
  CHAR_INFO		fill;

  if ( (actx = ConGetCursorX()) >= iNumOfCols - numchar ) {
    ConClearEOLine();
    return;
  } /* IF */

  from.Left = (SHORT)actx;
  from.Top = (SHORT)ConGetCursorY();
  from.Right = (SHORT)iNumOfCols;
  from.Bottom = from.Top;

  clip = from;
  to.X = (SHORT)(actx - numchar);
  to.Y = from.Top;

  fill.Char.AsciiChar = ' ';
  fill.Attributes = wAttributes; // WHICH ATTRIBUTES TO TAKE FOR BLANK CHAR ??

  ScrollConsoleScreenBuffer(hConsole, &from, &clip, to, &fill);
} /* DeleteCharacter */


void TTelnetParser::ConSetCursorPosition(int x, int y)
{
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
  if (x > CON_RIGHT)  x = CON_WIDTH;
  if (x < CON_LEFT)   x = CON_LEFT;
  if (y > CON_BOTTOM) y = CON_HEIGHT;
  if (y < CON_TOP)    y = CON_TOP;
	COORD Coord = {
		(short)(CON_LEFT + x),
		(short)(CON_TOP + y)};
	SetConsoleCursorPosition(hConsole, Coord);
}

int TTelnetParser::ConGetWidth(){
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return 0;
	return CON_COLS;
}
int TTelnetParser::ConGetHeight(){
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return 0;
	return CON_LINES;
}

int TTelnetParser::ConGetCursorX()
{
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return 0;
	return CON_CUR_X;
}

int TTelnetParser::ConGetCursorY()
{
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return 0;
	return CON_CUR_Y;
}

void TTelnetParser::ConMoveCursorPosition(int x, int y)
{
	CONSOLE_SCREEN_BUFFER_INFO ConsoleInfo;
	if (!GetConsoleScreenBufferInfo(hConsole, &ConsoleInfo))
		return;
	COORD Coord = {
		(SHORT) (CON_CUR_X + x),
		(SHORT) (CON_CUR_Y + y)};
	SetConsoleCursorPosition(hConsole, Coord);
}

void TTelnetParser::ConSetExtendedMode(int iFunction, BOOL bEnable)
{
	// Probably should do something here...
   // Should change the screen mode, but do we need this?
}

int TTelnetParser::NetWriteString(char* pszString, int cbString)
{
	send(Socket, pszString, cbString, 0);
	return cbString;
}