/*
VXDCHAIN.C
Contains Win32 functions to manipulate the Windows Virtual Device Driver
(VxD) chain.
Compile with -DSTANDALONE for Win32 Console app that displays the VxD chain

Andrew Schulman
andrew@ora.com
http://www.ora.com/windows/
ftp://ftp.ora.com/pub/examples/windows/win95.update/schulman.html
August 1995

The disk accompanying my book *Unauthorized Windows 95* (IDG Books,
1994) contained source code for a 16-bit protected-mode DOS version
of VXDCHAIN.  The "dangerous memory model" in Windows 95 (which, for
example, makes the entire VMM/VxD space read/write accessible to
every Win32 app -- so much for "private address spaces"!) makes it 
trivial to write Win32 apps for Win95 that access VxD code and data.

Sample output of standalone VXDCHAIN app (in this fairly typical
configuration, there were 74 VxDs installed):

Verified C000E990h by calling Get_VMM_DDB

Name      Vers   ID      DDB        Control    V86 API    PM API     #Srvc
--------  ----   -----   --------   --------   --------   --------   -----
VMM       4.00   0001h   C000E990   C00024F8   C0002C09   C0002C09*  402
SICE      1.00   0202h   C00465C0   C00206A9   C0020E94   C0020CB1   2
VCACHE    3.01   048Bh   C0083538   C0082F06   C00833E5   C00833E5   25
PERF      4.00   0048h   C008FD80   C008FC9C                         5
VPOWERD   4.00   0026h   C009F604   C009EF98   C02A76DC   C02A76DC*  14
VPICD     3.10   0003h   C00867D8   C008519C   C0085D30   C0085D30   25
VTD       4.00   0005h   C0087D44   C00879DD   C0226864   C0226864   11
VXDLDR    3.00   0027h   C0084604   C0084574   C0249F02   C0249F02   18
ISAPNP    4.00   003Ch   C0FD3780   C0FD3760                         0
PCCARD    4.00   097Ch   C0FDB80C   C0FD545E              C0FD5493*  2
CONFIGMG  4.00   0033h   C006BD6C   C006B9D0   C023F510   C023F510*  91
VCDFSD    3.00   0041h   C0084444   C00843D8                         4
MRCI2     3.10   0042h   C0FE1FD8   C0FE1FB8                         0
IOS       3.10   0010h   C00783E8   C00764E4   C027BD50   C027A2F0*  17
PAGEFILE  4.00   0021h   C00A0A84   C00A09E4              C022E59D*  10
PAGESWAP  2.10   0007h   C008C444   C008C3C4                         10
PARITY    1.00   0008h   C008EFA8   C008EEE8                         0
REBOOT    4.00   0009h   C00884B8   C0088250              C02269DC*  4
EBIOS     1.00   0012h   C009F70C   C009F6E4                         2
VDD       2.00   000Ah   C009FD08   C009F75C   C025375B   C025375B*  23
VSD       2.00   000Bh   C008B030   C008AE70                         4
COMBUFF   1.00           C006F2E4   C006F140                         0
VCD       3.10   000Eh   C00A226C   C00A1F18              C022E86C   13
VMOUSE    4.00   000Ch   C009ED20   C009E6E4   C02A4808   C02A4808*  12
VKD       2.10   000Dh   C00A34F0   C00A2334              C02559DC*  21
ENABLE    4.00   0037h   C00A3BD8   C00A38EA   C02A98AF   C02A9821*  10
LMOUSE    1.00   318Ah   C0FE90E8   C0FE7DA8   C11685A6   C11685A6   0
VPD       3.00   000Fh   C00A1348   C00A0ADC              C00A0D74   0
INT13     3.10   0020h   C00A1DC0   C00A1542                         5
VMCPD     4.00   0011h   C008F6D4   C008F09C              C008F0DD   9
BIOSXLAT  1.00   0013h   C008F048   C008F000                         0
VNETBIOS  3.00   0014h   C009E324   C009D27B                         8
NDIS      3.10   0028h   C0099C9C   C00961F3   C029C0F2   C029C0F2   96
PPPMAC    3.00   0499h   C0FEB9F4   C0FEB21C   C1177CCC*  C1177CCC*  10
ELPC3     3.10           C1008F24   C1009199                         0
NETBEUI   3.00   0031h   C10169F4   C100FCA8                         0
VTDI      3.00   0488h   C1019A20   C101960E                         14
WSOCK     1.00   003Eh   C106CA2C   C106C63F   C106C749*  C106C749*  5
VIP       3.00   0489h   C10267A0   C101AF5A   C101ADDB*             9
MSTCP     3.00   048Ah   C1030B20   C10278A0   C10278B4*             1
VDHCP     3.00   049Ah   C1034178   C1032773   C1032670*  C1032670*  4
VNBT      3.00   049Bh   C1049440   C1036672   C103686E*  C103686E*  1
AFVXD     0.00   0495h   C1072D18   C1072AA6                         3
DOSMGR    4.00   0015h   C008C640   C008C49C   C024F028              19
VSHARE    1.00   0483h   C006C1A8   C006C083   C006C058   C006C058   1
VMPOLL    4.00   0018h   C008DF7C   C008DE5C                         4
DSVXD     3.00   003Bh   C001AB94   C001A9C8   C001AA1F              0
SIWVID    1.00   7A5Fh   C001CFB4   C001AC00   C001AD54   C001AD54   4
VWIN32    1.02   002Ah   C006E6B0   C006D62C              C025DEB7*  29
VCOMM     1.00   002Bh   C006F000   C006ED0C   C0262BDC   C0262BDC*  35
VCOND     1.00   0038h   C008437C   C0084338   C027F48C*  C027F5B6*  2
VTDAPI    4.00   0442h   C008F9C4   C008F97D              C02865C4*  0
SAGE      0.46           C009E560   C009E6AA                         0
SOCKETSV  1.00           C0FD4D7C   C0FD3BE0                         0
DiskTSD   3.10           C0FDDA94   C0FDD83C                         0
voltrack  3.10   0090h   C0FE0F30   C0FE09F4                         0
DRVSPACX  3.10           C0FE310C   C0FE26F4                         0
RMM       3.10           C0FE5CD8   C0FE503C                         0
SPAP      1.00           C0FE967C   C0FE93B0   C0FE93FB*  C0FE940D*  0
ESDI_506  3.10   008Dh   C104D918   C104C600                         0
SERENUM   4.00           C0FD52AC   C0FD5284                         0
LPTENUM   4.00           C0FDE608   C0FDE098                         0
mmdevldr  1.00   044Ah   C10667DC   C106682C   C118A18C*  C118A18C*  6
WSHTCP    0.00           C10744A4   C1074438                         0
VDMAD     4.00   0004h   C008A348   C0088700                         34
V86MMGR   1.00   0006h   C008C23C   C008B61F              C022A070   25
SPOOLER   1.00   002Ch   C007882C   C0078748                         17
VFAT      3.00   0486h   C0082B70   C0080AE4                         0
VDEF      3.00           C0084D10   C0084AC4                         0
IFSMGR    3.00   0040h   C0074470   C006F470                         117
VNetSup   3.00   0480h   C006B88C   C006A810   C006A82D   C006A82D   7
VREDIR    3.00   0481h   C0094C94   C0093998                         17
VFINTD    3.01   2200h   C00A0974   C00A0743   C00A0756   C00A0756   0
SHELL     4.00   0017h   C008EAEC   C008E7DE   C0286595   C0286595*  28

To do:
Not tested in Win32s.  In Win32s, need BASE = 0x80000000L, not
0xC0000000L as in Win95.  Also, Win32 segment in Win32s doesn't have
base address of 0, so need to offset by 64K to access 0-based VxD
space??

Need to add in Win32 service table stuff
*/

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "vxdchain.h"
#include "excpt.h"

DWORD BASE = 0xC0000000L;
#define START (BASE + 0x1000L)      // don't start at BASE: page fault!
#define STOP  (BASE + 0x400000L)

#define VMM_STR     "VMM     "      // 5 spaces after name

// look for first occurrence of string inside block of memory
static char *memstr(char *fp, char *str, WORD len)
{
    int c = str[0];
    char *fp2 = fp;
    WORD len2 = len;
    _try {
        while (fp2 = (char *) memchr(fp2, c, len2))
        {
            if (strstr(fp2, str))
                return fp2;
            else
            {
                fp2++;  // skip past character
                len2 = len - (fp2 - fp);
            }
        }
    } _except (EXCEPTION_EXECUTE_HANDLER) { return NULL; }
    /* still here */
    return NULL;
}

int WINAPI Get_First_VxD(DWORD *proot)
{
    static DWORD vmm_ddb_lin = 0;
    DDB *pddb;
    char *vmm_str;  
    
    if (vmm_ddb_lin != 0) 
    {
        *proot = vmm_ddb_lin;
        return SUCCESS;
    }
    
    // try to find string "VMM     " in first 4MB of VMM/VxD space
    if (vmm_str = memstr((char *) START, VMM_STR, STOP - START))
    {
        // back up to get linear address of possible VMM_DDB
        vmm_ddb_lin = (DWORD) (vmm_str - offsetof(DDB, DDB_Name));
        pddb = (DDB *) vmm_ddb_lin;

        // make sure it really is VMM_DDB -- need more checking?
        if ((strncmp(pddb->DDB_Name, VMM_STR, 8) == 0) &&
            (pddb->DDB_Req_Device_Number == 1) &&
            (pddb->DDB_Init_Order == 0) &&  // VMM_Init_Order
            (pddb->DDB_Next > BASE) &&
            (pddb->DDB_Control_Proc > BASE))
        {
            // verify with (recursive) call to Get_VMM_DDB
            void (*Get_VMM_DDB)(void) = Get_VxD_Proc_Address(0x1013F);
            DWORD vmm_ddb;
            (*Get_VMM_DDB)();
            _asm mov vmm_ddb, eax
            if (vmm_ddb != vmm_ddb_lin)
                return ERROR_CANT_FIND_VMM;
#ifdef STANDALONE               
            printf("Verified %08lXh by calling Get_VMM_DDB\n", vmm_ddb);
#endif          
            *proot = vmm_ddb_lin;
            return SUCCESS;
        }
    }
    else
        return ERROR_CANT_FIND_VMM;
}

int WINAPI Get_Next_VxD(DDB *pddb, DWORD *pnext)
{
    DWORD next = pddb->DDB_Next;  // should verify with an Is_VxD() function?
    *pnext = next;
    return (next >= BASE) ? SUCCESS : ERROR_END_OF_CHAIN;
}

// should this be case-insensitive?
static int find_vxd(char *name, WORD id, DWORD *pvxd)
{
    DDB *pddb;
    DWORD vxd;
    int err;
    
    if ((err = Get_First_VxD(&vxd)) != 0)
        return err;
    for (;;)
    {
        pddb = (DDB *) vxd;
        if ((name && (strncmp(pddb->DDB_Name, name, 8) == 0)) ||
            (id && (pddb->DDB_Req_Device_Number == id)))
        {
            *pvxd = vxd;
            return SUCCESS;
        }
        if (Get_Next_VxD(pddb, &vxd) != SUCCESS)
            return ERROR_CANT_FIND_VXD;
    }
}

int WINAPI Find_VxD(char *name, DWORD *pvxd) { return find_vxd(name, 0, pvxd); }

int WINAPI Find_VxD_ID(WORD id, DWORD *pvxd) { return find_vxd((char *) 0,id,pvxd); }

FUNCPTR WINAPI Get_VxD_Proc_Address(DWORD func)
{
    WORD vxd_id = func >> 16;
    WORD func_num = func & 0xffff;
    DWORD vxd;
    FUNCPTR *srv_tab;
    DDB *ddb;
    if (Find_VxD_ID(vxd_id, &vxd) != SUCCESS)
        return 0;
    ddb = (DDB *) vxd;
    if (func_num > ddb->DDB_Service_Table_Size)
        return 0;
    srv_tab = (FUNCPTR *) ddb->DDB_Service_Table_Ptr;
    return srv_tab[func_num];
}

#ifdef STANDALONE

void fail(const char *s) { puts(s); exit(1); }

void spaces(int x) { int i; for (i=0; i<x; i++) putchar(' '); }

void display(DDB *pddb)
{
    unsigned char name[9];
    static int did_banner = 0;
    char *s;
    
    if (! did_banner)
    {
        did_banner = 1;
        printf("%s\n", "\n"
"Name      Vers   ID      DDB        Control    V86 API    PM API     #Srvc\n"
"--------  ----   -----   --------   --------   --------   --------   -----");
    }

    /* printf %8.8Fs should work, but doesn't seem to! */
    strncpy(name, pddb->DDB_Name, 8);
    name[8] = '\0';
    printf("%s  ", name);
    
    printf("%u.%02u   ", 
        pddb->DDB_Dev_Major_Version, pddb->DDB_Dev_Minor_Version);
    
    if (pddb->DDB_Req_Device_Number) 
        printf("%04Xh", pddb->DDB_Req_Device_Number);
    else
        spaces(5);
    spaces(3);
    
    printf("%08lX   ", pddb);
    printf("%08lX   ", pddb->DDB_Control_Proc);
    
    if (pddb->DDB_V86_API_Proc) printf("%08lX", pddb->DDB_V86_API_Proc);
    else spaces(8);
    printf( "%c", (pddb->DDB_V86_API_CSIP) ? '*' : ' ');
    spaces(2);
        
    if (pddb->DDB_PM_API_Proc) printf("%08lX", pddb->DDB_PM_API_Proc);
    else spaces(8);
    printf( "%c", (pddb->DDB_PM_API_CSIP) ? '*' : ' ');
    spaces(2);

    printf("%u", pddb->DDB_Service_Table_Size);
    if (pddb->DDB_Win32_Service_Table_Ptr)
    printf("\n");
}

int main(int argc, char *argv[])
{
    DWORD vxd_lin, next;
    int err;
    
    if ((err = Get_First_VxD(&vxd_lin)) != 0)
        fail("Can't locate VxD chain");
    for (;;)
    {
        display((DDB *) vxd_lin);
        if ((err = Get_Next_VxD((DDB *) vxd_lin, &next)) == 0)
            vxd_lin = next;
        else if (err == ERROR_END_OF_CHAIN)
            break;  // successfully got to end of VxD chain
        else
            fail("Can't walk VxD chain");
    }
    
    return 0;
}
#endif
