e/*
WALKWIN.C
Win32 Console app
Does recursive walk of window list, shows Win16 aspects of all windows.
Creates file WALKWIN.LOG

Andrew Schulman
andrew@ora.com
http://www.ora.com/windows/
ftp://ftp.ora.com/pub/examples/windows/win95.update/schulman.html
August 1995

*Unauthorized Windows 95* (IDG Books, 1994), pp. 453-454, includes a
Win16 version of this program.  Since the goal of the program is to
show that every Win32 window in Win95 has Win16 underpinnings, it's
more useful to make it a Win32 program.  The walkwin() function uses
the GetWindow() Win32 API call to do a straightforward recursive walk
of the window list.  However, the print_hwnd() function is far from
straightforward:

It calls GetWindowTask(), which returns an obfuscated Ring 3 thread
handle.  TIDToTDB() from UNOBFUSC.C is used to turn this into a pointer
to a Ring 3 Thread Control Block (THCB).  

From the THCB, print_hwnd() pulls out a selector to the underlying
Win16 Task Data Block (TDB; see *Undocumented Windows*, chapter 5).

This selector has to be turned into a flat pointer; this file
includes a Win32 version of GetSelectorBase() to do this.
GetSelectorBase() uses the undocumented VxDCall Win32 API to call DPMI
function 6.

Armed with a flat pointer to the Win16 TDB corresponding to the Win32
THCB corresponding to the HWND, print_hwnd() figures out if this is a 
Win16 or Win32 window.

cl walkwin.c k32exp.c unobfusc.c user32.lib

Sample output:
	// 16/32 HWND HTASK TDB16 Class Window
16 0564h  81529E60h 266Eh 'winio_wcmain'  "WINPSP"
16 0550h  8152944Ch 26D6h 'winio_wcmain'  "CallFunc"
32 04F4h  81526228h 1DF6h 'tooltips_class32'  
32 04E8h  81526228h 1DF6h 'CabinetWClass'  "My Computer"
  32 0500h  81526228h 1DF6h 'SHELLDLL_DefView'  
    32 0504h  81526228h 1DF6h 'SysListView32'  
      32 0508h  81526228h 1DF6h 'SysHeader32'  
  32 04ECh  81526228h 1DF6h 'msctls_statusbar32'  
  32 04F0h  81526228h 1DF6h 'ToolbarWindow32'  
    32 04F8h  81526228h 1DF6h 'ComboBox'  
32 0514h  81526228h 1DF6h 'ProxyTarget'  
32 0510h  81526228h 1DF6h 'ProxyTarget'  
?? 02E0h  DEAE6C0Bh 0000h 'ComboLBox'  
32 00ECh  81523944h 1DF6h 'ProxyTarget'  
...

WALKWIN while running Explorer, Internet Explorer, MS Exchange;
extracted window classes, ran through sort | uniq:

!Microsoft Exchange!               #32768
#32770                             #32771
#42                                BmpCC
Button                             CabinetWClass
CLIPBRDWNDCLASS 0x########         ComboBox
ComboLBox                          DdeCommonWindowClass 0x######## 
DMGClass                           DMGFrame
Edit                               ExploreWClass
Generic                            Internet Explorer_ANIMBTN
Internet Explorer_Frame            Internet Explorer_Hidden
Internet Explorer_HTML             LINEHVCLASS
ListBox                            MailViewBrowser
Microsoft Exchange 4.0 Viewer      MPCWndClass
msctls_progress32                  msctls_statusbar32
MSExtraPakWClass1                  MSTaskSwWClass
OleMainThreadWndClass 0x########   OTClass
Progman                            PROGRESSCLASS
ProxyTarget                        RICHEDIT
SAGEWINDOWCLASS                    SHELLDLL_DefView
Shell_TrayWnd                      SpoolProcessClass
Static                             SysHeader32
SysListView32                      SysTabControl32
SystemTray_Main                    SysTreeView32
ToolbarWindow                      ToolbarWindow32
tooltips_class                     tooltips_class32
TrayClockWClass                    TrayNotifyWnd
tty                                ttyGrab
WIN95 RPC Wmsg fffb923b            WIN95 RPC Wmsg fffc36ef
WIN95 RPC Wmsg fffedfdf            Windows 32-bit VxD Message Server
WMS Idle                           WMS notif engine
WMS Spooler                        WMS ST Notif Class
WMSUIVLB                           

These 60+ window classes are reasonable list of the "built-ins" for Win95??
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include "windows.h"
#include "unobfusc.h"
#include "k32exp.h"

unsigned num_hwnd = 0;
FILE *f;

void fail(const char *s) { puts(s); exit(1); }

DWORD GetSelectorBase(WORD sel)
{
    #define VXDCALL_ORD         1
    #define VWIN32_INT31_CALL   0x2A0029
    #define DPMICall(eax, ecx)  VxDCall(VWIN32_INT31_CALL, (eax), (ecx))
        
    static DWORD (WINAPI *VxDCall)(DWORD srvc, DWORD eax, DWORD ecx) =
        (DWORD (WINAPI *)(DWORD,DWORD,DWORD)) 0;
    if (! VxDCall)
        if (! (VxDCall = GetK32ProcAddress(VXDCALL_ORD)))
            return -1;
        
    _asm mov bx, sel
    DPMICall(0x0006, 0);    // DPMI Get Selector Base function
    // put CX:AX return value into EAX
    _asm mov ax, cx
    _asm shl eax, 16
    _asm mov ax, dx
}

void print_hwnd(HWND hwnd, int level)
{
    DWORD htask;
    WORD tdb16;
    DWORD tdb_flat;
    char *wndtext, *classname;
    int flag32;
    
    int i;
    
    if (! (wndtext = (char *) malloc(128)))
        fail("Insufficient memory");
    if (! (classname = (char *) malloc(128)))
        fail("Insufficient memory");
    
    // Win32 GetWindowTask returns obfuscated Ring 3 thread
    htask = TIDToTDB(GetWindowTask(hwnd));
    GetWindowText(hwnd, wndtext, 128);
    GetClassName(hwnd, classname, 128);
    _try {
        tdb16 = *((WORD *) (htask + 0x1C));  // Win16 TDB -- see CHGDIR.C
        if ((tdb_flat = GetSelectorBase(tdb16)) != -1)
            flag32 = *((WORD *) (tdb_flat + 0x16)) & 0x10;
        else
            flag32 = '?';
    } _except (EXCEPTION_EXECUTE_HANDLER) { flag32 = '?'; tdb16 = 0; }
    
    for (i=0; i<level; i++) {   // indent for each recursive level
        fprintf(f, "  ");
        printf("  ");
        }
	// 16/32 HWND HTASK TDB16 Class Window
    fprintf(f, "%s %04Xh  %04Xh %04Xh \'%s\'  ",
        flag32 == '?' ? "??" : flag32 ? "32" : "16",
        hwnd, htask, tdb16,
        classname);
    printf("%s %04Xh  %04Xh %04Xh \'%s\'  ",
        flag32 == '?' ? "??" : flag32 ? "32" : "16",
        hwnd, htask, tdb16,
        classname);
    if (wndtext && *wndtext) {
        fprintf(f, "\"%s\"", wndtext);
        printf("\"%s\"", wndtext);
        }
    fprintf(f, "\n");
    printf("\n");
    
    free(classname);
    free(wndtext);
}

void walkwin(HWND hwnd, int level)
{
    if (hwnd == 0)
        return;
    hwnd = GetWindow(hwnd, GW_HWNDFIRST);
    while (hwnd)
    {
        num_hwnd ++;
        print_hwnd(hwnd, level);
        walkwin(GetWindow(hwnd, GW_CHILD), level+1);
        hwnd = GetWindow(hwnd, GW_HWNDNEXT);
    }
    if (level == 0) { fprintf(f, "\n"); printf("\n"); }
}
    
main(int argc, char *argv[])
{
    HWND hwnd;
    if (! (f = fopen("walkwin.log", "w")))
        fail("Can't create WALKWIN.LOG");
    if (argc < 2)   hwnd = GetWindow(GetDesktopWindow(), GW_CHILD);
    else            sscanf(argv[1], "%04X", &hwnd);
    // for (;;)
    {
        num_hwnd = 0;
        walkwin(hwnd, 0);
        fprintf(f, "%u windows\n\n", num_hwnd);
        printf("%u windows\n\n", num_hwnd);
        fflush(f);
        // if (toupper(getchar()) == 'Q') break;
    }
    puts("Created WALKWIN.LOG");
    return 0;
}
